#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Divinity: Original Sin 2
# send your bug reports to bugs@dotslashplay.it
###

script_version=20250602.10

PLAYIT_COMPATIBILITY_LEVEL='2.32'

GAME_ID='divinity-original-sin-2'
GAME_NAME='Divinity: Original Sin 2'

GAME_ID_CLASSIC='divinity-original-sin-2-classic'
GAME_NAME_CLASSIC='Divinity: Original Sin 2 (classic)'

ARCHIVE_BASE_0_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811).exe'
ARCHIVE_BASE_0_MD5='bfb4abc5041d7cefd74931f97cfba847'
ARCHIVE_BASE_0_EXTRACTOR='innoextract'
## Do not convert file paths to lowercase, as it prevents some textures from being loaded.
ARCHIVE_BASE_0_EXTRACTOR_OPTIONS=' '
ARCHIVE_BASE_0_PART1_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-1.bin'
ARCHIVE_BASE_0_PART1_MD5='540a4f6433cfc4db5e7e2a9d225ceb66'
ARCHIVE_BASE_0_PART2_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-2.bin'
ARCHIVE_BASE_0_PART2_MD5='e6a3d1c21b8df032544b56ca9fe36c5a'
ARCHIVE_BASE_0_PART3_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-3.bin'
ARCHIVE_BASE_0_PART3_MD5='5548e2c4e1a83a4856af5e713afcd4d8'
ARCHIVE_BASE_0_PART4_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-4.bin'
ARCHIVE_BASE_0_PART4_MD5='04dd17ab5af1cd4dc1c8d06afd23ca6a'
ARCHIVE_BASE_0_PART5_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-5.bin'
ARCHIVE_BASE_0_PART5_MD5='4902d66b7e69d575a1ade4e20fb36381'
ARCHIVE_BASE_0_PART6_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-6.bin'
ARCHIVE_BASE_0_PART6_MD5='ea7452ece6e4652c97f2f552c711b184'
ARCHIVE_BASE_0_PART7_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-7.bin'
ARCHIVE_BASE_0_PART7_MD5='f9c8f89d3d2216da07eccf517b5ce406'
ARCHIVE_BASE_0_PART8_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-8.bin'
ARCHIVE_BASE_0_PART8_MD5='55f514d93f39bbf77f70c45f4ef532bb'
ARCHIVE_BASE_0_PART9_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-9.bin'
ARCHIVE_BASE_0_PART9_MD5='59f6aa38beaf00f738c5b96ff62f3459'
ARCHIVE_BASE_0_PART10_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-10.bin'
ARCHIVE_BASE_0_PART10_MD5='e5db62f0e3fe46abbbd9f289f6e5fe6a'
ARCHIVE_BASE_0_PART11_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-11.bin'
ARCHIVE_BASE_0_PART11_MD5='84b8f9a8595133d51e1572bd7bacfcb1'
ARCHIVE_BASE_0_PART12_NAME='setup_divinity_original_sin_2_-_definitive_edition_3.6.117.3735_ptbr_(58811)-12.bin'
ARCHIVE_BASE_0_PART12_MD5='60b7bdd7b74ed067588f209498688d42'
ARCHIVE_BASE_0_SIZE='62080538'
ARCHIVE_BASE_0_VERSION='3.6.117.3735-gog58811'
ARCHIVE_BASE_0_URL='https://www.gog.com/game/divinity_original_sin_2'

## Definitive Edition

CONTENT_PATH_DEFAULT='DefEd'
CONTENT_GAME_BIN_FILES='
bin'
CONTENT_GAME_L10N_EN_FILES='
data/localization/english.pak'
CONTENT_GAME_L10N_FR_FILES='
data/localization/french'
CONTENT_GAME_DATA_SOUNDS_FILES='
data/sharedsound*.pak
data/localization/voice*.pak'
CONTENT_GAME_DATA_VIDEOS_FILES='
data/localization/video*.pak'
CONTENT_GAME_DATA_TEXTURES_FILES='
data/textures*.pak'
CONTENT_GAME_DATA_FILES='
data/arena.pak
data/effects.pak
data/engine.pak
data/engineshaders.pak
data/gamemaster.pak
data/game.pak
data/gameplatform.pak
data/icons.pak
data/lowtex.pak
data/materials.pak
data/minimaps.pak
data/origins*.pak
data/patch*.pak
data/shared*.pak'

## Original build

CONTENT_PATH_DEFAULT_CLASSIC='.'
CONTENT_GAME_CLASSIC_BIN_FILES='
classic'
CONTENT_GAME_CLASSIC_L10N_EN_FILES='
data/localization/english.pak'
CONTENT_GAME_CLASSIC_L10N_FR_FILES='
data/localization/french.pak'
CONTENT_GAME_CLASSIC_DATA_SOUNDS_FILES='
data/sound
data/soundbanks.pak
data/localization/voice*.pak'
CONTENT_GAME_CLASSIC_DATA_VIDEOS_FILES='
data/localization/video*.pak'
CONTENT_GAME_CLASSIC_DATA_TEXTURES_1_FILES='
data/textures.pak
data/textures_[1-4].pak'
CONTENT_GAME_CLASSIC_DATA_TEXTURES_2_FILES='
data/textures_[5-9].pak'
CONTENT_GAME_CLASSIC_DATA_FILES='
data/arena.pak
data/effects.pak
data/engine.pak
data/engineshaders.pak
data/gamemaster.pak
data/game.pak
data/gameplatform.pak
data/icons.pak
data/lowtex.pak
data/materials.pak
data/minimaps.pak
data/shareddos.pak
data/squirrel.pak
data/origins*.pak
data/patch*.pak
data/shared*.pak'

WINE_PERSISTENT_DIRECTORIES='
users/${USER}/Documents/Larian Studios/Divinity Original Sin 2 Definitive Edition'
WINE_PERSISTENT_DIRECTORIES_CLASSIC='
users/${USER}/Documents/Larian Studios/Divinity Original Sin 2'
## With wined3d, some textures are rendered as plain white.
## cf. https://bugs.winehq.org/show_bug.cgi?id=43812
## (WINE 10.0)
WINE_DIRECT3D_RENDERER='dxvk'

APP_MAIN_EXE='bin/EoCApp.exe'
APP_MAIN_EXE_CLASSIC='Classic/EoCApp.exe'
## The game fails to display its window when using the wayland backend of SDL.
APP_MAIN_PRERUN='
# The game fails to display its window when using the wayland backend of SDL
if [ "${SDL_VIDEODRIVER:-}" = "wayland" ]; then
	unset SDL_VIDEODRIVER
fi
'

PACKAGES_LIST='
PKG_BIN
PKG_L10N_EN
PKG_L10N_FR
PKG_DATA_SOUNDS
PKG_DATA_VIDEOS
PKG_DATA_TEXTURES
PKG_DATA
PKG_CLASSIC_BIN
PKG_CLASSIC_L10N_EN
PKG_CLASSIC_L10N_FR
PKG_CLASSIC_DATA_SOUNDS
PKG_CLASSIC_DATA_VIDEOS
PKG_CLASSIC_DATA_TEXTURES_1
PKG_CLASSIC_DATA_TEXTURES_2
PKG_CLASSIC_DATA'

## Definitive Edition

PKG_BIN_ARCH='64'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_L10N
PKG_DATA'

PKG_L10N_ID="${GAME_ID}-l10n"
PKG_L10N_EN_ID="${PKG_L10N_ID}-en"
PKG_L10N_FR_ID="${PKG_L10N_ID}-fr"
PKG_L10N_PROVIDES="
$PKG_L10N_ID"
PKG_L10N_EN_PROVIDES="$PKG_L10N_PROVIDES"
PKG_L10N_FR_PROVIDES="$PKG_L10N_PROVIDES"
PKG_L10N_DESCRIPTION='localization'
PKG_L10N_EN_DESCRIPTION="$PKG_L10N_DESCRIPTION - English"
PKG_L10N_FR_DESCRIPTION="$PKG_L10N_DESCRIPTION - French"

PKG_DATA_ID="${GAME_ID}-data"
PKG_DATA_DESCRIPTION='data'
PKG_DATA_DEPENDENCIES_SIBLINGS='
PKG_DATA_SOUNDS
PKG_DATA_VIDEOS
PKG_DATA_TEXTURES'

PKG_DATA_SOUNDS_ID="${PKG_DATA_ID}-sounds"
PKG_DATA_SOUNDS_DESCRIPTION="$PKG_DATA_DESCRIPTION - sounds and voices"

PKG_DATA_VIDEOS_ID="${PKG_DATA_ID}-videos"
PKG_DATA_VIDEOS_DESCRIPTION="$PKG_DATA_DESCRIPTION - videos"

PKG_DATA_TEXTURES_ID="${PKG_DATA_ID}-textures"
PKG_DATA_TEXTURES_DESCRIPTION="$PKG_DATA_DESCRIPTION - textures"

## Original build

PKG_CLASSIC_BIN_ARCH='64'
PKG_CLASSIC_BIN_DEPENDENCIES_SIBLINGS='
PKG_CLASSIC_L10N
PKG_CLASSIC_DATA'

PKG_CLASSIC_L10N_ID="${GAME_ID_CLASSIC}-l10n"
PKG_CLASSIC_L10N_EN_ID="${PKG_CLASSIC_L10N_ID}-en"
PKG_CLASSIC_L10N_FR_ID="${PKG_CLASSIC_L10N_ID}-fr"
PKG_CLASSIC_L10N_PROVIDES="
$PKG_CLASSIC_L10N_ID"
PKG_CLASSIC_L10N_EN_PROVIDES="$PKG_CLASSIC_L10N_PROVIDES"
PKG_CLASSIC_L10N_FR_PROVIDES="$PKG_CLASSIC_L10N_PROVIDES"
PKG_CLASSIC_L10N_DESCRIPTION='localization'
PKG_CLASSIC_L10N_EN_DESCRIPTION="$PKG_CLASSIC_L10N_DESCRIPTION - English"
PKG_CLASSIC_L10N_FR_DESCRIPTION="$PKG_CLASSIC_L10N_DESCRIPTION - French"

PKG_CLASSIC_DATA_ID="${GAME_ID_CLASSIC}-data"
PKG_CLASSIC_DATA_DESCRIPTION='data'
PKG_CLASSIC_DATA_DEPENDENCIES_SIBLINGS='
PKG_CLASSIC_DATA_SOUNDS
PKG_CLASSIC_DATA_VIDEOS
PKG_CLASSIC_DATA_TEXTURES_1
PKG_CLASSIC_DATA_TEXTURES_2'

PKG_CLASSIC_DATA_SOUNDS_ID="${PKG_CLASSIC_DATA_ID}-sounds"
PKG_CLASSIC_DATA_SOUNDS_DESCRIPTION="$PKG_CLASSIC_DATA_DESCRIPTION - sounds and voices"

PKG_CLASSIC_DATA_VIDEOS_ID="${PKG_CLASSIC_DATA_ID}-videos"
PKG_CLASSIC_DATA_VIDEOS_DESCRIPTION="$PKG_CLASSIC_DATA_DESCRIPTION - videos"

PKG_CLASSIC_DATA_TEXTURES_ID="${PKG_CLASSIC_DATA_ID}-textures"
PKG_CLASSIC_DATA_TEXTURES_1_ID="${PKG_CLASSIC_DATA_TEXTURES_ID}-1"
PKG_CLASSIC_DATA_TEXTURES_2_ID="${PKG_CLASSIC_DATA_TEXTURES_ID}-2"
PKG_CLASSIC_DATA_TEXTURES_DESCRIPTION="$PKG_CLASSIC_DATA_DESCRIPTION - textures"
PKG_CLASSIC_DATA_TEXTURES_1_DESCRIPTION="$PKG_CLASSIC_DATA_TEXTURES_DESCRIPTION - 1"
PKG_CLASSIC_DATA_TEXTURES_2_DESCRIPTION="$PKG_CLASSIC_DATA_TEXTURES_DESCRIPTION - 2"

# Load common functions

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run the default initialization actions

initialization_default "$@"

# Extract game data

archive_extraction_default
(
	cd "${PLAYIT_WORKDIR}/gamedata/$(content_path_default)"

	# Definitive Edition - Generate the configuration of localization packages.
	config_source='Data/Localization/language.lsx'
	config_pattern='id="Value" value="[A-Za-z]*" type="20"'

	## English localization
	config_destination_en="$(package_path 'PKG_L10N_EN')$(path_game_data)/${config_source}"
	config_replacement_en='id="Value" value="English" type="20"'
	mkdir --parents "$(dirname "$config_destination_en")"
	sed --expression="s/${config_pattern}/${config_replacement_en}/" \
		"$config_source" > "$config_destination_en"

	## French localization
	config_destination_fr="$(package_path 'PKG_L10N_FR')$(path_game_data)/${config_source}"
	config_replacement_fr='id="Value" value="French" type="20"'
	mkdir --parents "$(dirname "$config_destination_fr")"
	sed --expression="s/${config_pattern}/${config_replacement_fr}/" \
		"$config_source" > "$config_destination_fr"
)
(
	cd "${PLAYIT_WORKDIR}/gamedata/$(
		set_current_package 'PKG_CLASSIC_BIN'
		content_path_default
	)"

	# Original build - Generate the configuration of localization packages.
	config_source='Data/Localization/language.lsx'
	config_pattern='id="Value" value="[A-Za-z]*" type="20"'

	## English localization
	config_destination_en="$(package_path 'PKG_CLASSIC_L10N_EN')$(set_current_package 'PKG_CLASSIC_L10N_EN';path_game_data)/${config_source}"
	config_replacement_en='id="Value" value="English" type="20"'
	mkdir --parents "$(dirname "$config_destination_en")"
	sed --expression="s/${config_pattern}/${config_replacement_en}/" \
		"$config_source" > "$config_destination_en"

	## French localization
	config_destination_fr="$(package_path 'PKG_CLASSIC_L10N_FR')$(set_current_package 'PKG_CLASSIC_L10N_FR';path_game_data)/${config_source}"
	config_replacement_fr='id="Value" value="French" type="20"'
	mkdir --parents "$(dirname "$config_destination_fr")"
	sed --expression="s/${config_pattern}/${config_replacement_fr}/" \
		"$config_source" > "$config_destination_fr"
)

# Include game data

content_inclusion_icons 'PKG_DATA'
## FIXME: Work around icons_inclusion_single_application incomplete support for package context.
##        (./play.it 2.32.4)
(
	set_current_package 'PKG_CLASSIC_DATA'
	content_inclusion_icons 'PKG_CLASSIC_DATA'
)
## FIXME: Work around content_inclusion incomplete support for package context.
##        (./play.it 2.32.4)
for package in $(packages_list); do
	(
		set_current_package "$package"
		content_inclusion_default_game_data "$package"
	)
done

# Write launchers

launchers_generation 'PKG_BIN'
launchers_generation 'PKG_CLASSIC_BIN'

# Build packages

packages_generation
case "$(messages_language)" in
	('fr')
		lang_string='Definitive Edition - version %s :'
		lang_en='anglaise'
		lang_fr='française'
	;;
	('en'|*)
		lang_string='Definitive Edition - %s version:'
		lang_en='English'
		lang_fr='French'
	;;
esac
printf '\n'
printf "$lang_string" "$lang_en"
print_instructions \
	'PKG_BIN' \
	'PKG_DATA' 'PKG_DATA_SOUNDS' 'PKG_DATA_VIDEOS' 'PKG_DATA_TEXTURES' \
	'PKG_L10N_EN'
printf "$lang_string" "$lang_fr"
print_instructions \
	'PKG_BIN' \
	'PKG_DATA' 'PKG_DATA_SOUNDS' 'PKG_DATA_VIDEOS' 'PKG_DATA_TEXTURES' \
	'PKG_L10N_FR'

case "$(messages_language)" in
	('fr')
		lang_string='Version originale - version %s :'
		lang_en='anglaise'
		lang_fr='française'
	;;
	('en'|*)
		lang_string='Original build - %s version:'
		lang_en='English'
		lang_fr='French'
	;;
esac
printf '\n'
printf "$lang_string" "$lang_en"
print_instructions \
	'PKG_CLASSIC_BIN' \
	'PKG_CLASSIC_DATA' 'PKG_CLASSIC_DATA_SOUNDS' 'PKG_CLASSIC_DATA_VIDEOS' 'PKG_CLASSIC_DATA_TEXTURES_1' 'PKG_CLASSIC_DATA_TEXTURES_2' \
	'PKG_CLASSIC_L10N_EN'
printf "$lang_string" "$lang_fr"
print_instructions \
	'PKG_CLASSIC_BIN' \
	'PKG_CLASSIC_DATA' 'PKG_CLASSIC_DATA_SOUNDS' 'PKG_CLASSIC_DATA_VIDEOS' 'PKG_CLASSIC_DATA_TEXTURES_1' 'PKG_CLASSIC_DATA_TEXTURES_2' \
	'PKG_CLASSIC_L10N_FR'

# Clean up

working_directory_cleanup

exit 0
