/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gbemol-app.h"
#include "gbemol-utils.h"
#include "gbemol-preferences.h"
#include "bacon-volume.h"

typedef struct _GbemolThreadData {
	GbemolApp* app;
	GbemolMpdSong* song;
} GbemolThreadData;

struct _GbemolAppPrivate {
	GdkPixbuf* icon;
	GtkStatusIcon* status_icon;

	GtkWidget *scl_song;
	gboolean update_scl_song;

	GtkWidget *btn_previous;
	GtkWidget *btn_play;
	GtkWidget *btn_next;
	GtkWidget *btn_connect;
	GtkWidget *btn_volume;

	GtkWidget *statusbar;
	GtkWidget *img_cover;

	GtkWidget *btn_artist;
	GtkWidget *btn_album;
	GtkWidget *btn_title;
	GtkWidget *lbl_time;
	GtkWidget *lbl_dash;
	GtkWidget *lbl_from;

	GtkWidget *mnu_status;
	GtkWidget *img_item_play;

	GtkWidget *mnu_main;

	GtkWidget *expander;

	GtkWidget *win_main;
	GtkWidget *mnu_item_random;
	GtkWidget *mnu_item_repeat;
	GtkWidget *mnu_item_disconnect;
	GtkWidget *mnu_item_connect;

	GtkTooltips *tips;
	
	GThreadPool* pool;
	GbemolPreferences *pref;
};

static GObjectClass *parent_class = NULL;

static const gchar* authors[] = {"Felipe Weckx <felipe.weckx@gmail.com>", 
				NULL};

/* Class Functions */
static void gbemol_app_class_init (GbemolApp *g_class);
static void gbemol_app_init (GbemolApp *obj);
static void gbemol_app_finalize (GObject *object);
static void gbemol_app_dispose (GObject *object);
static void gbemol_app_build_menus (GbemolApp* app);
static void gbemol_app_build_gui (GbemolApp* app);

static gboolean gbemol_app_win_main_resize_idled (GbemolApp* app);

/* GTK+ Callbacks */
static gboolean on_win_main_delete_event (GtkWidget *w, GdkEventButton *event, GbemolApp* data);
static void on_btn_previous_clicked (GtkButton *button, GbemolApp* app);
static void on_btn_play_clicked (GtkButton *button, GbemolApp* app);
static void on_btn_next_clicked (GtkButton *button, GbemolApp* app);
static void on_btn_title_clicked (GtkButton *button, GbemolApp* app);
static void on_btn_album_clicked (GtkButton *button, GbemolApp* app);
static void on_btn_artist_clicked (GtkButton *button, GbemolApp* app);
static gboolean on_evt_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolApp* data);
static void on_mnu_random_activate (GtkWidget* w, GbemolApp* data);
static void on_mnu_repeat_activate (GtkWidget* w, GbemolApp* data);
static void on_mnu_about_activate (GtkWidget* w, GbemolApp* data);
static void on_mnu_connect_activate (GtkWidget* b, GbemolApp* data);
static void on_expander_activate (GtkExpander* e, GbemolApp* data);
static void on_mnu_quit_activate (GtkWidget* e, GbemolApp* data);
static void on_mnu_preferences_activate (GtkWidget* e, GbemolApp* data);
static gboolean on_scl_song_change_value (GtkRange* range, GtkScrollType* scroll, gdouble value, GbemolApp* data);
static void on_status_icon_popup_menu (GtkStatusIcon* icon, guint button, guint activate_time, GbemolApp* app);
static void on_status_icon_activate (GtkStatusIcon *icon, GbemolApp* app);
static void on_btn_volume_value_changed (BaconVolumeButton *btn, GbemolApp* app);
static void on_notebook_page_switch (GtkNotebook* n, GtkNotebookPage* p, gint page_num, GbemolApp* app);
static gpointer gbemol_app_set_cover_image(GbemolThreadData* data);


GType
gbemol_app_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolAppClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc) gbemol_app_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolApp),
			0,      /* n_preallocs */
			(GInstanceInitFunc) gbemol_app_init    /* instance_init */
		};

		type = g_type_register_static (G_TYPE_OBJECT,
				"GbemolApp",
				&info, 0);
	}
	return type;
}

static void
gbemol_app_build_menus (GbemolApp* app)
{
	GtkWidget *item, *img;


	/*** MAIN MENU ***/
	app->priv->mnu_main = gtk_menu_new ();

	/* About */
	item = gtk_image_menu_item_new_with_label (_("About"));
	img = gtk_image_new_from_stock (GTK_STOCK_ABOUT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_about_activate), app);
	gtk_widget_show (item);

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	gtk_widget_show (item);

	/* Random */
	item = gtk_check_menu_item_new_with_mnemonic (_("R_andom"));
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_random_activate), app);
	gtk_widget_show (item);
	app->priv->mnu_item_random = item;

	/* Repeat */
	item = gtk_check_menu_item_new_with_mnemonic (_("_Repeat"));
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_repeat_activate), app);
	gtk_widget_show (item);
	app->priv->mnu_item_repeat = item;

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	gtk_widget_show (item);

	/* Preferences */
	item = gtk_image_menu_item_new_with_mnemonic (_("_Preferences"));
	img = gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_preferences_activate), app);
	gtk_widget_show (item);

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	gtk_widget_show (item);

	/* Connect/Disconnect */
	item = gtk_image_menu_item_new_with_mnemonic (_("_Connect"));
	img = gtk_image_new_from_stock (GTK_STOCK_CONNECT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_connect_activate), app);
	gtk_widget_show (item);
	app->priv->mnu_item_connect = item;

	item = gtk_image_menu_item_new_with_mnemonic (_("_Disconnect"));
	img = gtk_image_new_from_stock (GTK_STOCK_DISCONNECT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_connect_activate), app);
	gtk_widget_show (item);
	app->priv->mnu_item_disconnect = item;

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	gtk_widget_show (item);

	/* Quit */
	item = gtk_image_menu_item_new_with_label (_("Quit"));
	img = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_main), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_quit_activate), app);
	gtk_widget_show (item);


	/*** STATUS ICON MENU ****/
	app->priv->mnu_status = gtk_menu_new ();

	/* Next */
	item = gtk_image_menu_item_new_with_label (_("Next"));
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_status), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_btn_next_clicked), app);
	gtk_widget_show (item);

	/* Play/Pause */
	item = gtk_image_menu_item_new_with_label (_("Play/Pause"));
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	app->priv->img_item_play = img;
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_status), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_btn_play_clicked), app);
	gtk_widget_show (item);

	/* Previous */
	item = gtk_image_menu_item_new_with_label (_("Previous"));
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_status), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_btn_previous_clicked), app);
	gtk_widget_show (item);

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_status), item);
	gtk_widget_show (item);

	/* Quit */
	item = gtk_image_menu_item_new_with_label (_("Quit"));
	img = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (app->priv->mnu_status), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_quit_activate), app);
	gtk_widget_show (item);
}

static void
gbemol_app_build_gui (GbemolApp *obj)               
{
	GtkWidget *hbox1, *hbox2, *vbox1, *vbox2, *ebox, *btn, *lbl, *img;

	obj->priv->tips = gtk_tooltips_new ();

	obj->priv->icon = gdk_pixbuf_new_from_file (ICON_FILE, NULL);

	gbemol_app_build_menus (obj);

	/* Main window */
	obj->priv->win_main = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_icon (GTK_WINDOW (obj->priv->win_main), obj->priv->icon);
	if (gbemol_cfg_get_bool("win_main", "stick"))
		gtk_window_stick (GTK_WINDOW (obj->priv->win_main));
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->win_main), 5);
	g_signal_connect (G_OBJECT (obj->priv->win_main), "delete-event", G_CALLBACK (on_win_main_delete_event), obj);

	/* Status icon */
	if (gbemol_cfg_get_bool("app", "status-icon"))
		gbemol_app_create_status_icon (obj);

	ebox = gtk_event_box_new ();
	g_signal_connect (G_OBJECT (ebox), "button-press-event", G_CALLBACK (on_evt_button_press_event), obj);
	gtk_container_add (GTK_CONTAINER (obj->priv->win_main), ebox);

	/* The main vbox */
	vbox1 = gtk_vbox_new (FALSE, 2);
	gtk_container_add (GTK_CONTAINER (ebox), vbox1);

	/* Top HBox */
	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);
	
	/* Left-corner VBox */
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox2, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);

	/* Artist */
	btn = gtk_button_new_with_label ("Artist");
	obj->priv->btn_artist = btn;
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);  
	gtk_button_set_alignment (GTK_BUTTON (btn), 0.0, 0.5);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("More songs from this artist"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_artist_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_artist_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox2), btn, FALSE, FALSE, 0);

	lbl = gtk_label_new ("");	
	gtk_label_set_markup (GTK_LABEL (lbl), _("<span size=\"large\" weight=\"ultrabold\">-</span>"));
	gtk_box_pack_start (GTK_BOX (hbox2), lbl, FALSE, FALSE, 0);
	obj->priv->lbl_dash = lbl;

	/* Song Title */
	btn = gtk_button_new_with_label ("Title");
	obj->priv->btn_title = btn;
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);  
	gtk_button_set_alignment (GTK_BUTTON (btn), 0.0, 0.5);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("Song Information"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_title_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_title_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox2), btn, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);

	lbl = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (lbl), _("<span size=\"small\">from</span>"));
	gtk_box_pack_start (GTK_BOX (hbox2), lbl, FALSE, FALSE, 0);
	obj->priv->lbl_from = lbl;

	/* Album */
	btn = gtk_button_new_with_label ("Album");
	obj->priv->btn_album = btn;
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);  
	gtk_button_set_alignment (GTK_BUTTON (btn), 0.0, 0.5);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("More songs from this album"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_album_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_album_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox2), btn, FALSE, FALSE, 0);

	/* Cover image */
	obj->priv->img_cover = gtk_image_new ();
	gtk_misc_set_alignment (GTK_MISC (obj->priv->img_cover), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->img_cover, TRUE, TRUE, 0);

	/* Player hbox */
	hbox1 = gtk_hbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, TRUE, 0);

	/* Previous */
	btn = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_BUTTON);
	gtk_button_set_image (GTK_BUTTON (btn), img);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("Previous Song"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_previous_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_previous_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	obj->priv->btn_previous = btn;
	
	/* Play/Pause */
	btn = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON);
	gtk_button_set_image (GTK_BUTTON (btn), img);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("Play"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_play_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_play_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	obj->priv->btn_play = btn;

	/* Next */
	btn = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_BUTTON);
	gtk_button_set_image (GTK_BUTTON (btn), img);
	gtk_tooltips_set_tip (obj->priv->tips, btn, _("Next Song"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_next_clicked), obj);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_next_clicked), obj);
	gtk_box_pack_start (GTK_BOX (hbox1), btn, FALSE, FALSE, 0);
	obj->priv->btn_next = btn;

	/* Scrollbar */
	obj->priv->scl_song = gtk_hscale_new_with_range (0.0, 1.0, 0.01);
	gtk_scale_set_draw_value (GTK_SCALE (obj->priv->scl_song), FALSE);
	gtk_range_set_update_policy (GTK_RANGE (obj->priv->scl_song), GTK_UPDATE_DISCONTINUOUS);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->scl_song, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (obj->priv->scl_song), "change-value", G_CALLBACK (on_scl_song_change_value), obj);

	/* Time Label */
	obj->priv->lbl_time = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->lbl_time, FALSE, FALSE, 0);

	/* Volume Button */
	obj->priv->btn_volume = bacon_volume_button_new (GTK_ICON_SIZE_BUTTON, 0.0, 100.0, 1.0);
	gtk_button_set_relief (GTK_BUTTON (obj->priv->btn_volume), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->btn_volume, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (obj->priv->btn_volume), "value-changed", 
			G_CALLBACK (on_btn_volume_value_changed), obj);

	/* Expander */
	obj->priv->expander = gtk_expander_new (NULL);
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->expander), 3);
	gtk_box_pack_start (GTK_BOX (vbox1), obj->priv->expander, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (obj->priv->expander), "activate", G_CALLBACK (on_expander_activate), obj);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (obj->priv->expander), vbox2);

	/* Notebook */
	obj->notebook = gtk_notebook_new ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (obj->notebook), GTK_POS_LEFT);
	gtk_box_pack_start (GTK_BOX (vbox2), obj->notebook, TRUE, TRUE, 5);

	hbox1 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox1, FALSE, FALSE, 0);

	/* Status bar 
	obj->priv->statusbar = gtk_statusbar_new ();
	gtk_box_pack_start (GTK_BOX (hbox1), obj->priv->statusbar, TRUE, TRUE, 0);
	*/

	gtk_widget_show_all (ebox);

	if (gbemol_cfg_get_int ("win_main", "expanded"))
		gtk_expander_set_expanded (GTK_EXPANDER (obj->priv->expander), TRUE);

	g_idle_add ((GSourceFunc) gbemol_app_win_main_resize_idled, obj);
}

static void
gbemol_app_init (GbemolApp* app)
{
	gchar *host, *pass, *str;
	gint i, defaults[] = {0, 1, 2, 10};
	GtkWidget* lbl;
	GtkNotebook* n;	

	app->priv = g_new0 (GbemolAppPrivate, 1);
	app->mpd = NULL;
	app->priv->status_icon = NULL;
	app->priv->update_scl_song = TRUE;
	app->pls = NULL;
	app->lib = NULL;
	app->info = NULL;	
	
	/* Build Interface */
	gbemol_app_build_gui (app);

	/* Check if there is a config file */
	if (!(host = gbemol_cfg_get_string ("MPD", "host")))
	{
		app->mpd = gbemol_mpd_new_with_defaults ();

		/* Save default columns in playlist and library, otherwise nothing is shown */
		for (i = 0; i < 4; i++)
		{
			if (i == 3)
				str = g_strdup_printf ("column%d", defaults[i] - 1);
			else
				str = g_strdup_printf ("column%d", defaults[i]);
			gbemol_cfg_set_int ("library", str, 1);
			g_free (str);

			str = g_strdup_printf ("column%d", defaults[i] + 1);
			gbemol_cfg_set_int ("playlist", str, 1);			
			g_free (str);
		}
	} 
	else 
	{
		/* We need a password? */
		if (gbemol_cfg_get_int ("MPD", "use_auth"))
			pass = gbemol_cfg_get_string ("MPD", "pass");
		else
			pass = NULL;

		app->mpd = gbemol_mpd_new (host, pass, gbemol_cfg_get_int ("MPD", "port"), 
				gbemol_cfg_get_double ("MPD", "timeout"));
		g_free (host);
		g_free (pass);
	}

	n = GTK_NOTEBOOK (app->notebook);
	g_signal_connect (G_OBJECT (n), "switch-page", G_CALLBACK (on_notebook_page_switch), app);

	/* Playlist Tab */
	app->pls = gbemol_playlist_new (app->mpd);
	lbl = gtk_label_new (_("Playlist"));
	gtk_label_set_angle (GTK_LABEL (lbl), 90.0);
	gtk_notebook_append_page (n, GTK_WIDGET (app->pls), lbl);

	/* Library Tab */
	app->lib = gbemol_library_new (app->mpd);
	lbl = gtk_label_new (_("Library"));
	gtk_label_set_angle (GTK_LABEL (lbl), 90.0);
	gtk_notebook_append_page (n, GTK_WIDGET (app->lib), lbl);

	/* Info Tab */
	app->info = gbemol_info_new (app->mpd);
	lbl = gtk_label_new (_("Info"));
	gtk_label_set_angle (GTK_LABEL (lbl), 90.0);
	gtk_notebook_append_page (n, GTK_WIDGET (app->info), lbl);	
}

static void
gbemol_app_class_init (GbemolApp* klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->dispose = gbemol_app_dispose;
	gobject_class->finalize = gbemol_app_finalize;

  	parent_class = g_type_class_peek_parent (klass);
}

void
gbemol_app_finalize (GObject *obj)
{

	gtk_widget_destroy (GBEMOL_APP (obj)->priv->win_main);
	g_object_unref (G_OBJECT (GBEMOL_APP (obj)->mpd));
	/*g_thread_pool_free(GBEMOL_APP(obj)->priv->pool, TRUE, TRUE);*/
	g_free (GBEMOL_APP (obj)->priv);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

GbemolApp* gbemol_app_new (void)
{
	GbemolApp* obj;

	obj = GBEMOL_APP (g_object_new (GBEMOL_TYPE_APP, NULL));

	return obj;
}

void gbemol_app_set_mpd (GbemolApp* obj, GbemolMpd* mpd)
{
	obj->mpd = mpd;
}

void gbemol_app_create_status_icon (GbemolApp* obj)
{
	if (!obj->priv->status_icon)
	{
		obj->priv->status_icon = gtk_status_icon_new_from_file (ICON_FILE);
		g_signal_connect (G_OBJECT (obj->priv->status_icon), "popup-menu", 
				G_CALLBACK (on_status_icon_popup_menu), obj);
		g_signal_connect (G_OBJECT (obj->priv->status_icon), "activate", 
				G_CALLBACK (on_status_icon_activate), obj);
	}
}

void gbemol_app_destroy_status_icon (GbemolApp* obj)
{
	g_object_unref (G_OBJECT (obj->priv->status_icon));	
	obj->priv->status_icon = NULL;
}

void gbemol_app_connect_mpd (GbemolApp* app)
{
	if (!app->mpd)
		return;

	if (gbemol_cfg_get_bool("MPD", "use_auth"))
		gbemol_mpd_connect_and_authenticate (app->mpd);
	else
		gbemol_mpd_connect (app->mpd);

	gbemol_app_set_connected (app, gbemol_mpd_is_connected (app->mpd));
	if (gbemol_mpd_is_connected (app->mpd))
	{
		gbemol_library_start_constraints (app->lib);
		gbemol_playlist_update_songs (app->pls);
		gbemol_playlist_populate_playlists (app->pls);
	}
}
	

static GtkWidget* 
gbemol_app_create_about_dialog (GbemolApp* app)
{
	GtkWidget* dlg;

	dlg = gtk_about_dialog_new ();

	gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (dlg), "gbemol");
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (dlg), GBEMOL_VERSION);
	gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (dlg), authors);
	gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (dlg), "http://gbemol.sourceforge.net");
	gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (dlg), app->priv->icon);
	gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (dlg), _("translator-credits"));
	gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (dlg), _("A GTK+ Interface to MPD"));

	return dlg;
}

void 
gbemol_app_destroy (GbemolApp* obj)
{
	
}

void 
gbemol_app_dispose (GObject* obj)
{

}

static gpointer gbemol_app_set_cover_image(GbemolThreadData* data)
{
	GdkPixbuf *pix_img = NULL, *scaled;	
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
	
	gdk_threads_enter();
    g_static_mutex_lock (&mutex);
	
	pix_img = gbemol_utils_get_song_image(data->song);
	if (pix_img)
	{
		scaled = gdk_pixbuf_scale_simple (pix_img, 70, 70, GDK_INTERP_BILINEAR);
		gtk_image_set_from_pixbuf (GTK_IMAGE (data->app->priv->img_cover), scaled);
		g_object_unref(pix_img);
		g_object_unref(scaled);
		gtk_widget_show(data->app->priv->img_cover);
	}
	else
		gtk_widget_hide(data->app->priv->img_cover);
	
	gbemol_mpd_free_song(data->song);
	g_free(data);

	g_static_mutex_unlock (&mutex);
	gdk_threads_leave();
	
	return NULL;
}

void
gbemol_app_set_song (GbemolApp* app, GbemolMpdSong* song)
{
	gchar *str_title, *str_artist = NULL, *str_album = NULL;
	GtkWidget *lbl;
		
	if (!song)
	{	    
		str_title = g_markup_printf_escaped (
				_("<span weight=\"ultrabold\" size=\"large\">Not Playing</span>"));
		gtk_window_set_title (GTK_WINDOW (app->priv->win_main), "gbemol");
		if (app->priv->status_icon)
			gtk_status_icon_set_tooltip (app->priv->status_icon, _("Not Playing"));

		gtk_widget_hide (app->priv->btn_artist);
		gtk_widget_hide (app->priv->lbl_dash);
		gtk_widget_hide (app->priv->lbl_from);
		gtk_widget_hide (app->priv->btn_album);
	}

	else
	{		
		/* Window title */
		if (gbemol_mpd_song_is_stream (song))
			str_title = g_strdup_printf ("%s - %s", song->name, song->title);
		else
			str_title = g_strdup_printf ("%s - %s", song->artist, song->title);

		gtk_window_set_title (GTK_WINDOW (app->priv->win_main), str_title);

		if (app->priv->status_icon)
			gtk_status_icon_set_tooltip (app->priv->status_icon, str_title);

		g_free (str_title);

		/* It's a stream */
		if (gbemol_mpd_song_is_stream (song))
		{
			str_title = g_markup_printf_escaped ("<span weight=\"heavy\" size=\"large\">%s</span>", 
					song->title);
			str_artist = NULL;
			str_album = g_markup_printf_escaped ("<span size=\"small\" weight=\"bold\">%s</span>", 
					song->name);

			gtk_widget_hide (app->priv->btn_artist);
			gtk_widget_hide (app->priv->lbl_dash);
		}
		else
		{
			str_album = g_markup_printf_escaped ("<span size=\"small\" weight=\"bold\">%s</span>", 
					song->album);
			str_title = g_markup_printf_escaped ("<span weight=\"heavy\" size=\"large\">%s</span>",
				       song->title);
			str_artist = g_markup_printf_escaped ("<span size=\"large\" weight=\"bold\">%s</span>", 
					song->artist);
			
			gtk_widget_show (app->priv->btn_artist);
			gtk_widget_show (app->priv->lbl_dash);		
			gtk_widget_show (app->priv->lbl_from);
			gtk_widget_show (app->priv->btn_album);

			GbemolThreadData* data = g_malloc(sizeof(GbemolThreadData));
			data->app = app;
			data->song = gbemol_mpd_songdup(song);

			g_thread_create((GThreadFunc)gbemol_app_set_cover_image, (gpointer) data, FALSE, NULL);

		}

	}

	lbl = gtk_bin_get_child (GTK_BIN (app->priv->btn_title));
	gtk_label_set_markup (GTK_LABEL (lbl), str_title);
	lbl = gtk_bin_get_child (GTK_BIN (app->priv->btn_artist));
	gtk_label_set_markup (GTK_LABEL (lbl), str_artist);
	lbl = gtk_bin_get_child (GTK_BIN (app->priv->btn_album));
	gtk_label_set_markup (GTK_LABEL (lbl), str_album);

	/* Set the info tab */
	g_free (str_title);
	g_free (str_artist);
	g_free (str_album);
		
}

void gbemol_app_set_state (GbemolApp* app)
{
	GtkWidget *img;

	if (gbemol_mpd_is_connected (app->mpd))
	{
		switch (gbemol_mpd_get_state(app->mpd))
		{
			case MPD_STATUS_STATE_PLAY:
				img = gtk_image_new_from_stock ("gtk-media-pause", GTK_ICON_SIZE_BUTTON);
				gtk_button_set_image (GTK_BUTTON (app->priv->btn_play), img);
				gtk_image_set_from_stock (GTK_IMAGE (app->priv->img_item_play), "gtk-media-pause",
						GTK_ICON_SIZE_BUTTON);
				gtk_tooltips_set_tip (app->priv->tips, app->priv->btn_play, _("Pause"), NULL);
				gtk_widget_set_sensitive (app->priv->scl_song, TRUE);				
				gtk_widget_set_sensitive (app->priv->lbl_time, TRUE);
				break;				
			default:
				if (gbemol_mpd_get_state(app->mpd) != MPD_STATUS_STATE_PAUSE)
					gtk_range_set_value (GTK_RANGE (app->priv->scl_song), 0.0);
				img = gtk_image_new_from_stock ("gtk-media-play", GTK_ICON_SIZE_BUTTON);
				gtk_button_set_image (GTK_BUTTON (app->priv->btn_play), img);
				gtk_image_set_from_stock (GTK_IMAGE (app->priv->img_item_play),"gtk-media-play", GTK_ICON_SIZE_BUTTON);
				gtk_tooltips_set_tip (app->priv->tips, app->priv->btn_play, _("Play"), NULL);
				gtk_widget_set_sensitive (app->priv->scl_song, FALSE);
				gtk_widget_set_sensitive (app->priv->lbl_time, FALSE);
				gtk_label_set_text (GTK_LABEL (app->priv->lbl_time), "00:00");
				break;			
		}
	}
}

void
gbemol_app_set_connected (GbemolApp* app, gboolean connected)
{
	GbemolMpdSong* song;

	gtk_widget_set_sensitive (app->priv->mnu_item_connect, !connected);
	gtk_widget_set_sensitive (app->priv->mnu_item_disconnect, connected);

	if (connected)
	{
		song = gbemol_mpd_get_current_song (app->mpd);
		gbemol_app_set_song (app, song);
		gbemol_info_set_song (app->info, song);
		gbemol_mpd_free_song (song);
		gbemol_app_set_state (app);
	}
	else
		gbemol_app_set_song (app, NULL);

	gbemol_playlist_set_sensitive (app->pls, connected);
	gbemol_library_set_sensitive (app->lib, connected);

	gtk_widget_set_sensitive (app->priv->btn_previous, connected);
	gtk_widget_set_sensitive (app->priv->btn_play, connected);
	gtk_widget_set_sensitive (app->priv->btn_next, connected);
	gtk_widget_set_sensitive (app->priv->btn_volume, connected);
	gtk_widget_set_sensitive (app->priv->btn_artist, connected);
	gtk_widget_set_sensitive (app->priv->btn_album, connected);
	gtk_widget_set_sensitive (app->priv->btn_title, connected);
	gtk_widget_set_sensitive (app->priv->mnu_item_random, connected);
	gtk_widget_set_sensitive (app->priv->mnu_item_repeat, connected);
	gtk_widget_set_sensitive (app->notebook, connected);
}

void
gbemol_app_set_time (GbemolApp* app, int elapsed, int total)
{
	gchar *str;
	double prog;

	str = g_markup_printf_escaped ("<span weight=\"ultrabold\">%02d:%02d/%02d:%02d</span>", 
			elapsed/60, elapsed%60, total / 60, total % 60);
	gtk_label_set_markup (GTK_LABEL (app->priv->lbl_time), str);
	g_free (str);

	prog = ((double) elapsed) / ((double) total);
	if (app->priv->update_scl_song)
		gtk_range_set_value (GTK_RANGE (app->priv->scl_song), prog);
	app->priv->update_scl_song = !app->priv->update_scl_song;

}

void gbemol_app_set_volume (GbemolApp* app, gint volume)
{
	if ((volume >= 0) && (volume <= 100))
		bacon_volume_button_set_value (BACON_VOLUME_BUTTON (app->priv->btn_volume), (float) volume);
}

void gbemol_app_set_random (GbemolApp* app, gboolean random)
{
	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (app->priv->mnu_item_random)) != random)
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (app->priv->mnu_item_random), random);
}

void gbemol_app_set_repeat (GbemolApp* app, gboolean repeat)
{
	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (app->priv->mnu_item_repeat)) != repeat)
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (app->priv->mnu_item_repeat), repeat);
}

void gbemol_app_set_stick (GbemolApp* app, gboolean stick)
{
	if (stick)
		gtk_window_stick (GTK_WINDOW (app->priv->win_main));
	else
		gtk_window_unstick (GTK_WINDOW (app->priv->win_main));
}

void gbemol_app_set_top (GbemolApp* app, gboolean top)
{
	gtk_window_set_keep_above (GTK_WINDOW (app->priv->win_main), top);
}

void gbemol_app_show (GbemolApp* app)
{	
	gtk_window_present(GTK_WINDOW(app->priv->win_main));
}

void gbemol_app_hide (GbemolApp* app)
{
	gtk_widget_hide (app->priv->win_main);
}

/*
 * Switches the app to full mode
 */
void gbemol_app_full_mode (GbemolApp* app)
{
	gtk_expander_set_expanded (GTK_EXPANDER (app->priv->expander), TRUE);
}

/*
 * Switch to compact mode
 */
void gbemol_app_compact_mode (GbemolApp* app)
{

	gtk_expander_set_expanded (GTK_EXPANDER (app->priv->expander), FALSE);
}

/* Callbacks */
static gboolean on_win_main_delete_event (GtkWidget *w, GdkEventButton *event, GbemolApp* data)
{
	if (gbemol_cfg_get_bool("app", "close-tray") && gbemol_cfg_get_bool("app", "status-icon"))
	{
		gtk_widget_hide (data->priv->win_main);
		gbemol_cfg_set_bool("win_main", "visible", TRUE);
		return TRUE;
	}
	else
	{
		g_object_unref (G_OBJECT(data));
		gbemol_app_destroy (data);
		gtk_main_quit ();
		return FALSE;
	}
}

static void on_btn_previous_clicked (GtkButton *button, GbemolApp* app)
{
	gbemol_mpd_player_previous (app->mpd);
}


static void on_btn_play_clicked (GtkButton *button, GbemolApp* app)
{
	gbemol_mpd_player_pause (app->mpd);
}

static void on_btn_next_clicked (GtkButton *button, GbemolApp* app)
{
	gbemol_mpd_player_next (app->mpd);
}

static gboolean on_evt_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolApp* app)
{
	if (event->type == GDK_BUTTON_PRESS)
		if (event->button == 3)
		{
			gtk_menu_popup (GTK_MENU (app->priv->mnu_main), 
					NULL, NULL, NULL, NULL, 
					event->button, event->time);
			return TRUE;
		}

	return FALSE;
}

static void on_mnu_random_activate (GtkWidget* w, GbemolApp* app)
{
	gboolean active;

	active = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(w));
	gbemol_mpd_set_random (app->mpd, active);
}

static void on_mnu_repeat_activate (GtkWidget* w, GbemolApp* app)
{
	gboolean active;

	active = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(w));
	gbemol_mpd_set_repeat (app->mpd, active);
}

static void on_mnu_about_activate (GtkWidget* w, GbemolApp* app)
{
	GtkWidget* dlg;
	dlg = gbemol_app_create_about_dialog (app);

	gtk_widget_show (dlg);
	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);
}

static void 
on_mnu_connect_activate (GtkWidget* b, GbemolApp* app)
{
	if (gbemol_mpd_is_connected (app->mpd))
		gbemol_mpd_disconnect (app->mpd);
	else
		gbemol_app_connect_mpd (app);

	gbemol_app_set_connected (app, gbemol_mpd_is_connected (app->mpd));
}

static gboolean gbemol_app_win_main_resize_idled (GbemolApp* app)
{
	gint height, width, x, y;

	if (gbemol_cfg_get_int ("win_main", "expanded"))
	{
		height = gbemol_cfg_get_int ("win_main", "height_big");
		width = gbemol_cfg_get_int ("win_main", "width_big");
		x = gbemol_cfg_get_int ("win_main", "x_big");
		y = gbemol_cfg_get_int ("win_main", "y_big");
	}
	else
	{
		height = gbemol_cfg_get_int ("win_main", "height_small");
		width = gbemol_cfg_get_int ("win_main", "width_small");
		x = gbemol_cfg_get_int ("win_main", "x_small");
		y = gbemol_cfg_get_int ("win_main", "y_small");
		gtk_window_unmaximize (GTK_WINDOW (app->priv->win_main));
	}

	if (!height)
		height = GBEMOL_DEFAULT_HEIGHT;
	if (!width)
		width = GBEMOL_DEFAULT_WIDTH;


	gtk_widget_set_size_request (app->priv->win_main, width, height);
	gtk_window_resize (GTK_WINDOW (app->priv->win_main), width, height);
	gtk_widget_set_size_request (app->priv->win_main, 0, 0);
	
	gtk_window_move(GTK_WINDOW(app->priv->win_main), x, y);
		
	return FALSE;	
}

static void on_expander_activate (GtkExpander* e, GbemolApp* app)
{
	gint h_s, h_b, width, x, y;

	h_s = gbemol_cfg_get_int ("win_main", "height_small");
	h_b = gbemol_cfg_get_int ("win_main", "height_big");
	
	gtk_window_get_position(GTK_WINDOW(app->priv->win_main), &x, &y);
	if (!gtk_expander_get_expanded (e))
	{
		gtk_window_get_size (GTK_WINDOW (app->priv->win_main), &width, &h_s);
		gbemol_cfg_set_int ("win_main", "height_small", h_s);
		gbemol_cfg_set_int ("win_main", "width_small", width);
		gbemol_cfg_set_int ("win_main", "x_small", x);
		gbemol_cfg_set_int ("win_main", "y_small", y);
		gbemol_cfg_set_int ("win_main", "expanded", 1);

		if (!h_b)
		{
			h_b = h_s * 4;
			gbemol_cfg_set_int ("win_main", "height_big", h_b);
		}
	}
	else
	{	
		/* Save sizes */
		gtk_window_get_size (GTK_WINDOW (app->priv->win_main), &width, &h_b);
		gbemol_cfg_set_int ("win_main", "height_big", h_b);
		gbemol_cfg_set_int ("win_main", "width_big", width);
		gbemol_cfg_set_int ("win_main", "x_big", x);
		gbemol_cfg_set_int ("win_main", "y_big", y);
		gbemol_cfg_set_int ("win_main", "expanded", 0);
	}

	g_idle_add ((GSourceFunc) gbemol_app_win_main_resize_idled, app);
}


static gboolean on_scl_song_change_value (GtkRange* range, GtkScrollType* scroll, gdouble value, GbemolApp* app)
{
	gchar* str;
	gint t;

	app->priv->update_scl_song = FALSE;

	t = (gint) (gbemol_mpd_get_total_time (app->mpd) * value);
	if (t < 0)
		return FALSE;
	
	str = g_markup_printf_escaped ("%02d:%02d", t / 60, t % 60);
	gtk_label_set_markup (GTK_LABEL (app->priv->lbl_time), str);
	g_free (str);

	gbemol_mpd_seek (app->mpd, gbemol_mpd_get_current_id (app->mpd), t);

	app->priv->update_scl_song = TRUE;
	
	return FALSE;
}

static void on_mnu_quit_activate (GtkWidget* m, GbemolApp* app)
{
	g_object_unref (G_OBJECT (app));
	gtk_main_quit ();
}
			
static void on_mnu_preferences_activate (GtkWidget* m, GbemolApp* data)
{
	gbemol_preferences_new (data);
}

static void 
on_status_icon_popup_menu (GtkStatusIcon* icon, guint button, guint activate_time, GbemolApp* app)
{
	GtkMenu* mnu = GTK_MENU (app->priv->mnu_status);

	gtk_menu_popup (mnu, NULL, NULL, gtk_status_icon_position_menu, icon, button, activate_time);
}

static void 
on_status_icon_activate (GtkStatusIcon *icon, GbemolApp* app)
{
	if (GTK_WIDGET_VISIBLE (app->priv->win_main))
		gtk_widget_hide (app->priv->win_main);
	else
		gtk_widget_show (app->priv->win_main);
	gbemol_cfg_set_int ("win_main", "visible", !GTK_WIDGET_VISIBLE (app->priv->win_main));
}
static void on_btn_volume_value_changed (BaconVolumeButton *btn, GbemolApp* app)
{
	gbemol_mpd_set_volume (app->mpd, (int) bacon_volume_button_get_value (btn));
}

static void on_btn_title_clicked (GtkButton *button, GbemolApp* app)
{
	GbemolMpdSong *song = gbemol_mpd_get_current_song (app->mpd);
	/* It's a stream, do nothing */
	if (!song->artist && song->name && (song->time == MPD_SONG_NO_TIME))
		return;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (app->notebook), 2);
	gbemol_info_set_song (app->info, song);
	gbemol_mpd_free_song (song);
}

static void on_btn_album_clicked (GtkButton *button, GbemolApp* app)
{
	GbemolMpdSong *song = gbemol_mpd_get_current_song (app->mpd);

	/* It's a stream, do nothing */
	if (!song->artist && song->name && (song->time == MPD_SONG_NO_TIME))
		return;

	gtk_notebook_set_current_page (GTK_NOTEBOOK (app->notebook), 2);
	gbemol_info_set_album (app->info, song->album, song->artist);
	gbemol_mpd_free_song (song);
}

static void on_btn_artist_clicked (GtkButton *button, GbemolApp* app)
{
	GbemolMpdSong *song = gbemol_mpd_get_current_song (app->mpd);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (app->notebook), 2);
	gbemol_info_set_artist (app->info, song->artist);
	gbemol_mpd_free_song (song);

}

static void on_notebook_page_switch (GtkNotebook* n, GtkNotebookPage* p, gint page_num, GbemolApp* app)
{
}

void gbemol_app_error_dialog_new (GbemolApp* app, gchar* primary_msg, gchar* secondary_msg)
{
	GtkWidget* dlg;
	gchar *str1 = NULL, *str2 = NULL;

	if (primary_msg)
		str1 = g_markup_printf_escaped ("<span size=\"large\" weight=\"bold\">%s</span>", primary_msg);
	if(secondary_msg)
		str2 = g_strdup (secondary_msg);

	dlg = gtk_message_dialog_new (GTK_WINDOW (app->priv->win_main), 
			GTK_DIALOG_DESTROY_WITH_PARENT,
                        GTK_MESSAGE_ERROR,
                        GTK_BUTTONS_CLOSE,
                        NULL);

	if (str1)
		gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dlg), str1);
	if (str2)
		gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dlg), str2);

	gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);

	g_free (str1);
	g_free (str2);
}

static void on_btn_reconnect_activate (GtkButton *button, GtkDialog* dlg)
{
	gtk_dialog_response (dlg, GTK_RESPONSE_ACCEPT);
}


void gbemol_app_reconnect_dialog_new (GbemolApp* app, gchar* msg)
{
	GtkWidget* dlg, *btn;
	gchar *str1 = NULL, *str2 = NULL;

	str1 = g_markup_printf_escaped (_("<span size=\"large\" weight=\"bold\">Disconnected</span>"));
	str2 = g_strdup (msg);

	btn = gbemol_utils_make_button (_("Reconnect"), GTK_STOCK_REFRESH);
	dlg = gtk_message_dialog_new (GTK_WINDOW (app->priv->win_main), 
			GTK_DIALOG_DESTROY_WITH_PARENT,
                        GTK_MESSAGE_ERROR,
                        GTK_BUTTONS_CLOSE,
                        NULL);
	gtk_box_pack_end (GTK_BOX (GTK_DIALOG(dlg)->action_area), btn, FALSE, FALSE, 5);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_reconnect_activate), dlg);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_reconnect_activate), dlg);
	gtk_widget_show_all (btn);

	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dlg), str1);
	gtk_message_dialog_format_secondary_markup (GTK_MESSAGE_DIALOG (dlg), str2);

	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_ACCEPT)
		gbemol_app_connect_mpd (app);

	gtk_widget_destroy (dlg);
	g_free (str1);
	g_free (str2);
}



