/**
 * @file libedit_plot_component.cpp
 */

/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 1992-2018 KiCad Developers, see AUTHORS.txt for contributors.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you may find one here:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * or you may search the http://www.gnu.org website for the version 2 license,
 * or you may write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */


#include <fctsys.h>

#include <gr_basic.h>
#include <pgm_base.h>
#include <sch_draw_panel.h>
#include <confirm.h>
#include <gestfich.h>
#include <eeschema_id.h>
#include <sch_screen.h>
#include <wildcards_and_files_ext.h>

#include <general.h>
#include <lib_edit_frame.h>
#include <class_library.h>
#include <dialogs/dialog_plot_schematic.h>


void LIB_EDIT_FRAME::OnPlotCurrentComponent( wxCommandEvent& event )
{
    wxString   fullFileName;
    wxString   file_ext;
    wxString   mask;

    LIB_PART*      part = GetCurPart();

    if( !part )
    {
        wxMessageBox( _( "No component" ) );
        return;
    }

    switch( event.GetId() )
    {
    case ID_LIBEDIT_GEN_PNG_FILE:
    {
        mask = wxT( "*." ) + file_ext;
        wxFileName fn( part->GetName() );
        fn.SetExt( "png" );

        wxString projectPath = wxPathOnly( Prj().GetProjectFullName() );

        wxFileDialog dlg( this, _( "Image File Name" ), projectPath,
                          fn.GetFullName(), PngFileWildcard(), wxFD_SAVE | wxFD_OVERWRITE_PROMPT );

        if( dlg.ShowModal() == wxID_CANCEL || dlg.GetPath().IsEmpty() )
            return;

        // calling wxYield is mandatory under Linux, after closing the file selector dialog
        // to refresh the screen before creating the PNG or JPEG image from screen
        wxYield();
        CreateImageFile( dlg.GetPath(), wxBITMAP_TYPE_PNG );
    }
        break;

    case ID_LIBEDIT_GEN_SVG_FILE:
    {
        file_ext = wxT( "svg" );
        mask     = wxT( "*." ) + file_ext;
        wxFileName fn( part->GetName() );
        fn.SetExt( file_ext );

        wxString pro_dir = wxPathOnly( Prj().GetProjectFullName() );

        fullFileName = EDA_FILE_SELECTOR( _( "Filename:" ), pro_dir,
                                          fn.GetFullName(), file_ext, mask, this,
                                          wxFD_SAVE, true );

        if( fullFileName.IsEmpty() )
            return;

        PAGE_INFO pageSave = GetScreen()->GetPageSettings();
        PAGE_INFO pageTemp = pageSave;

        wxSize componentSize = part->GetUnitBoundingBox( m_unit, m_convert ).GetSize();

        // Add a small margin to the plot bounding box
        pageTemp.SetWidthMils(  int( componentSize.x * 1.2 ) );
        pageTemp.SetHeightMils( int( componentSize.y * 1.2 ) );

        GetScreen()->SetPageSettings( pageTemp );
        SVG_PlotComponent( fullFileName );
        GetScreen()->SetPageSettings( pageSave );
    }
        break;
    }
}


void LIB_EDIT_FRAME::CreateImageFile( const wxString& aFileName, wxBitmapType aBitmapType )
{
    if( !saveCanvasImageToFile( aFileName, aBitmapType ) )
    {
        wxString msg;
        msg.Printf( _( "Can't save file \"%s\"." ), aFileName );
        wxMessageBox( msg );
    }
}


void LIB_EDIT_FRAME::SVG_PlotComponent( const wxString& aFullFileName )
{
    const bool plotColor = true;
    const PAGE_INFO& pageInfo = GetScreen()->GetPageSettings();

    SVG_PLOTTER* plotter = new SVG_PLOTTER();
    plotter->SetPageSettings( pageInfo );
    plotter->SetDefaultLineWidth( GetDefaultLineThickness() );
    plotter->SetColorMode( plotColor );

    wxPoint plot_offset;
    const double scale = 1.0;

    // Currently, plot units are in decimil
    plotter->SetViewport( plot_offset, IU_PER_MILS/10, scale, false );

    // Init :
    plotter->SetCreator( wxT( "Eeschema-SVG" ) );

    if( ! plotter->OpenFile( aFullFileName ) )
    {
        delete plotter;
        return;
    }

    LOCALE_IO   toggle;

    plotter->StartPlot();

    LIB_PART*      part = GetCurPart();

    if( part )
    {
        TRANSFORM   temp;     // Uses default transform
        wxPoint     plotPos;

        plotPos.x = pageInfo.GetWidthIU() /2;
        plotPos.y = pageInfo.GetHeightIU()/2;

        part->Plot( plotter, GetUnit(), GetConvert(), plotPos, temp );

        // Plot lib fields, not plotted by m_component->Plot():
        part->PlotLibFields( plotter, GetUnit(), GetConvert(), plotPos, temp );
    }

    plotter->EndPlot();
    delete plotter;
}


void LIB_EDIT_FRAME::PrintPage( wxDC* aDC, LSET aPrintMask, bool aPrintMirrorMode, void* aData )
{
    LIB_PART*      part = GetCurPart();

    if( !part )
        return;

    wxSize pagesize = GetScreen()->GetPageSettings().GetSizeIU();

    /* Plot item centered to the page
     * In libedit, the component is centered at 0,0 coordinates.
     * So we must plot it with an offset = pagesize/2.
     */
    wxPoint plot_offset;
    plot_offset.x = pagesize.x / 2;
    plot_offset.y = pagesize.y / 2;

    part->Draw( m_canvas, aDC, plot_offset, m_unit, m_convert, PART_DRAW_OPTIONS::Default() );
}
