#include <QDomDocument>
#include "MainWindow.h"
#include "RenderArea.h"
#include "Graph.h"

Graph::Graph(QWidget* parent) : QWidget(parent)
{
    mainWindow = static_cast<MainWindow*>(parent);
    setMinimumWidth(400);
    setAttribute(Qt::WA_DeleteOnClose);
    scaleStyle = Graph::Decimal;

    gridLayout = new QGridLayout;
    QGridLayout* outerGridLayout = new QGridLayout(this);

    y1LineEdit = new QLineEdit(this);
    y2LineEdit = new QLineEdit(this);
    y3LineEdit = new QLineEdit(this);
    y1LineEdit->installEventFilter(this);
    y2LineEdit->installEventFilter(this);
    y3LineEdit->installEventFilter(this);
    xMinLineEdit =  new QLineEdit("-5.0", this);
    xMaxLineEdit =  new QLineEdit("5.0", this);
    yMinLineEdit =  new QLineEdit("-3.0", this);
    yMaxLineEdit =  new QLineEdit("3.0", this);
    renderArea = new RenderArea(this);

    gridLayout->addWidget(new QLabel(tr("<font color='blue'>y1 = </font>"), this), 0, 0);
    gridLayout->addWidget(new QLabel(tr("<font color='red'>y2 = </font>"), this), 1, 0);
    gridLayout->addWidget(new QLabel(tr("<font color='green'>y3 = </font>"), this), 2, 0);
    gridLayout->addWidget(y1LineEdit, 0, 1);
    gridLayout->addWidget(y2LineEdit, 1, 1);
    gridLayout->addWidget(y3LineEdit, 2, 1);

    gridLayout->addWidget(new QLabel(tr("   X min "), this), 0, 2);
    gridLayout->addWidget(new QLabel(tr("   X max "), this), 0, 4);
    gridLayout->addWidget(xMinLineEdit, 0, 3);
    gridLayout->addWidget(xMaxLineEdit, 0, 5);

    gridLayout->addWidget(new QLabel(tr("   Y min "), this), 1, 2);
    gridLayout->addWidget(new QLabel(tr("   Y max "), this), 1, 4);
    gridLayout->addWidget(yMinLineEdit, 1, 3);
    gridLayout->addWidget(yMaxLineEdit, 1, 5);

    gridLayout->setColumnStretch(1, 1);
    gridLayout->setColumnMinimumWidth(1, 100);
    gridLayout->setContentsMargins(2, 2, 2, 2);
    gridLayout->setMargin(10);
    gridLayout->setSpacing(4);
    outerGridLayout->setSpacing(0);
    outerGridLayout->addLayout(gridLayout, 0, 0);
    outerGridLayout->addWidget(renderArea, 1, 0);
    outerGridLayout->setRowMinimumHeight(1, 200);
    outerGridLayout->setContentsMargins(0, 0, 0, 0);
    outerGridLayout->setRowStretch(1, 1);

    setLayout(outerGridLayout);
}

Graph::~Graph()
{
    //    delete currentFile;
}

void Graph::plot()
{
    qreal xMin = mainWindow->evaluate(0, xMinLineEdit->text()).toReal();
    qreal yMin = mainWindow->evaluate(0, yMinLineEdit->text()).toReal();
    qreal xMax = mainWindow->evaluate(0, xMaxLineEdit->text()).toReal();
    qreal yMax = mainWindow->evaluate(0, yMaxLineEdit->text()).toReal();

    xMinOrig = xMinLineEdit->text();
    xMaxOrig = xMaxLineEdit->text();
    yMinOrig = yMinLineEdit->text();
    yMaxOrig = yMaxLineEdit->text();

    renderArea->logicalRectange.setTopLeft(QPointF(xMin, yMin));
    renderArea->logicalRectange.setBottomRight(QPointF(xMax, yMax));
    renderArea->plot();
    renderArea->repaint();
}

void Graph::zoomIn()
{
    qreal l = renderArea->logicalRectange.left();
    qreal r = renderArea->logicalRectange.right();
    qreal t = renderArea->logicalRectange.top();
    qreal b = renderArea->logicalRectange.bottom();
    renderArea->logicalRectange.setLeft((3*l+r)/4);
    renderArea->logicalRectange.setRight((3*r+l)/4);
    renderArea->logicalRectange.setTop((3*t+b)/4);
    renderArea->logicalRectange.setBottom((3*b+t)/4);
    repopulateParameterEdits();
    renderArea->plot();
    window->setWindowModified(true);
}

void Graph::zoomOut()
{
    qreal l = renderArea->logicalRectange.left();
    qreal r = renderArea->logicalRectange.right();
    qreal t = renderArea->logicalRectange.top();
    qreal b = renderArea->logicalRectange.bottom();
    renderArea->logicalRectange.setLeft((3*l-r)/2);
    renderArea->logicalRectange.setRight((3*r-l)/2);
    renderArea->logicalRectange.setTop((3*t-b)/2);
    renderArea->logicalRectange.setBottom((3*b-t)/2);
    repopulateParameterEdits();
    renderArea->plot();
    window->setWindowModified(true);
}

void Graph::repopulateParameterEdits()
{
    xMinLineEdit->setText(QString::number(renderArea->logicalRectange.left()));
    xMaxLineEdit->setText(QString::number(renderArea->logicalRectange.right()));
    yMinLineEdit->setText(QString::number(renderArea->logicalRectange.top()));
    yMaxLineEdit->setText(QString::number(renderArea->logicalRectange.bottom()));
}

void Graph::zoomReset()
{
    xMinLineEdit->setText(xMinOrig);
    xMaxLineEdit->setText(xMaxOrig);
    yMinLineEdit->setText(yMinOrig);
    yMaxLineEdit->setText(yMaxOrig);
    plot();
}

bool Graph::eventFilter(QObject *object, QEvent *event)
{
    if ((object == y1LineEdit || object == y2LineEdit || object == y3LineEdit) && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent*>(event);
        if (keyEvent->key() == Qt::Key_Return) {
            plot();
            window->setWindowModified(true);
        }
    }
    return QWidget::eventFilter(object, event);
}

bool Graph::openFile(QString fileName)
{    
    QFile file(fileName);
    if (!file.open(QFile::ReadOnly | QFile::Text))
        return false;

    QDomDocument doc;
    bool ok = doc.setContent(&file);
    file.close();

    if (!ok) {
        QMessageBox::critical(this, tr("File Open Error"), tr("Problem with file format"));
        return false;
    }

    QDomElement root = doc.documentElement();
    if (root.tagName() != "graph") {
        QMessageBox::critical(this, tr("Lybniz"), tr("The file is not a Lybniz file."));
        return false;
    }

    window->setWindowTitle(QFileInfo(file).fileName() + "[*]");
    window->setWindowFilePath(QFileInfo(file).absoluteFilePath());
    QDomNode e = root.firstChildElement("parameters");
    y1LineEdit->setText(e.firstChildElement("y1").text());
    y2LineEdit->setText(e.firstChildElement("y2").text());
    y3LineEdit->setText(e.firstChildElement("y3").text());
    xMinLineEdit->setText(e.firstChildElement("xmin").text());
    yMinLineEdit->setText(e.firstChildElement("ymin").text());
    xMaxLineEdit->setText(e.firstChildElement("xmax").text());
    yMaxLineEdit->setText(e.firstChildElement("ymax").text());

    e = root.firstChildElement("settings");
    QString s = e.firstChildElement("geometry").text();
    QStringList sl = s.split(",");
    window->resize(QSize(sl[2].toInt(), sl[3].toInt()));
    window->setWindowModified(false);
    return true;
}

bool Graph::saveAsFile()
{
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save Function Definitions"), window->windowFilePath(), "Lybniz (*.lyb)");
    if (fileName.isNull())
        return false;

    QFileInfo fi = QFileInfo(fileName);
    window->setWindowTitle(fi.fileName());
    window->setWindowFilePath(fi.absoluteFilePath());
    return saveFile();
}

bool Graph::saveFile()
{
    if (window->windowFilePath().isEmpty())
        return saveAsFile();

    QDomDocument doc("Lybniz");
    QDomElement root = doc.createElement("graph");

    QDomElement e = doc.createElement("parameters");
    root.appendChild(e);

    QDomElement tag = doc.createElement("y1");
    e.appendChild(tag);
    QDomText t = doc.createTextNode(y1LineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("y2");
    e.appendChild(tag);
    t = doc.createTextNode(y2LineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("y3");
    e.appendChild(tag);
    t = doc.createTextNode(y3LineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("xmin");
    e.appendChild(tag);
    t = doc.createTextNode(xMinLineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("xmax");
    e.appendChild(tag);
    t = doc.createTextNode(xMaxLineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("ymin");
    e.appendChild(tag);
    t = doc.createTextNode(yMinLineEdit->text());
    tag.appendChild(t);

    tag = doc.createElement("ymax");
    e.appendChild(tag);
    t = doc.createTextNode(yMaxLineEdit->text());
    tag.appendChild(t);

    e = doc.createElement("settings");
    root.appendChild(e);

    QString s;
    tag = doc.createElement("geometry");
    e.appendChild(tag);
    s = QString("%1,%2,%3,%4").arg(window->pos().x()).arg(window->pos().y()).arg(window->size().width()).arg(window->size().height());
    t = doc.createTextNode(s);
    tag.appendChild(t);

    tag = doc.createElement("scale_style");
    e.appendChild(tag);
    t = doc.createTextNode(QString::number(1));
    tag.appendChild(t);

    tag = doc.createElement("connect_points");
    e.appendChild(tag);
    t = doc.createTextNode(renderArea->connectPoints ? "true" : "false");
    tag.appendChild(t);

    tag = doc.createElement("logical_rectange");
    e.appendChild(tag);
    s = QString("%1,%2,%3,%4").arg(renderArea->logicalRectange.left()).arg(renderArea->logicalRectange.top()).arg(renderArea->logicalRectange.right()).arg(renderArea->logicalRectange.bottom());
    t = doc.createTextNode(s);
    tag.appendChild(t);

    doc.appendChild(root);

    //    qDebug() << window->geometry()<< pos().x()<< renderArea->logicalRectange->top();


    QFile file(window->windowFilePath());

    if (!file.open(QFile::WriteOnly | QFile::Text))
        return false;

    QTextStream stream(&file);
    stream.setCodec("UTF-8");
    stream << doc.toString(4);

    //    file.writeData(doc.toString(4));
    file.close();
    window->setWindowModified(false);
    return true;
}

void Graph::close()
{
    window->close();
}

bool Graph::maybeSave()
{
    if (window->isWindowModified()) {
        QMessageBox::StandardButton ret;
        ret = QMessageBox::warning(this, tr("Lybniz"), tr("'%1' has been modified.\nDo you want to save your changes?").arg(window->windowTitle()),
                                   QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        if (ret == QMessageBox::Save)
            return saveFile();
        else if (ret == QMessageBox::Cancel)
            return false;
    }
    return true;
}

void Graph::closeEvent(QCloseEvent *event)
{
    if (maybeSave()) {
        event->accept();
    } else {
        event->ignore();
    }
}
