/*	canvas.c
	Copyright (C) 2004-2007 Mark Tyler and Dmitry Groshev

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "global.h"

#include "memory.h"
#include "png.h"
#include "mainwindow.h"
#include "otherwindow.h"
#include "mygtk.h"
#include "canvas.h"
#include "viewer.h"
#include "toolbar.h"


float can_zoom = 1;				// Zoom factor 1..MAX_ZOOM
int margin_main_x, margin_main_y,		// Top left of image from top left of canvas
	margin_view_x, margin_view_y;
int zoom_flag;
int marq_status = MARQUEE_NONE,
	marq_x1 = -1, marq_y1 = -1, marq_x2 = -1, marq_y2 = -1;		// Selection marquee
int marq_drag_x, marq_drag_y;						// Marquee dragging offset

gboolean text_paste = FALSE,			// Are we pasting text?
	close_after_save = FALSE		// Closedown after save
	;


void commit_paste( gboolean undo )
{
	int fx, fy, fw, fh, fx2, fy2;		// Screen coords
	int i, ofs = 0, ua;
	unsigned char *image, *mask, *alpha = NULL;

	fx = marq_x1 > 0 ? marq_x1 : 0;
	fy = marq_y1 > 0 ? marq_y1 : 0;
	fx2 = marq_x2 < mem_width ? marq_x2 : mem_width - 1;
	fy2 = marq_y2 < mem_height ? marq_y2 : mem_height - 1;

	fw = fx2 - fx + 1;
	fh = fy2 - fy + 1;

	ua = 0;//channel_dis[CHN_ALPHA];	// Ignore clipboard alpha if disabled

	mask = malloc(fw);
	if (!mask) return;	/* !!! Not enough memory */
	if ((mem_channel == CHN_IMAGE) /*&& RGBA_mode*/ && !mem_clip_alpha &&
		!ua && mem_img[CHN_ALPHA])
	{
		alpha = malloc(fw);
		if (!alpha) return;
//		memset(alpha, channel_col_A[CHN_ALPHA], fw);
	}
	ua |= !mem_clip_alpha;

	if ( undo ) mem_undo_next(UNDO_PASTE);	// Do memory stuff for undo

	/* Offset in memory */
	if (marq_x1 < 0) ofs -= marq_x1;
	if (marq_y1 < 0) ofs -= marq_y1 * mem_clip_w;
	image = mem_clipboard + ofs * mem_clip_bpp;

	for (i = 0; i < fh; i++)
	{
		row_protected(fx, fy + i, fw, mask);
		paste_pixels(fx, fy + i, fw, mask, image, ua ?
			alpha : mem_clip_alpha + ofs, mem_clip_mask ?
			mem_clip_mask + ofs : NULL, 255);
		image += mem_clip_w * mem_clip_bpp;
		ofs += mem_clip_w;
	}

	free(mask);
	free(alpha);

	update_menus();				// Update menu undo issues
//	vw_update_area(fx, fy, fw, fh);
	main_update_area(fx, fy, fw, fh);
}

void paste_prepare()
{
	if ( tool_type != TOOL_SELECT )
	{
		clear_perim();
		toolbar_select_select_tool();
	}
	else
	{
		if ( marq_status != MARQUEE_NONE ) paint_marquee(0, marq_x1, marq_y1);
	}
}

void paste_init()
{
	marq_status = MARQUEE_PASTE;
	cursor_corner = -1;
	update_menus();
	gtk_widget_queue_draw( drawing_canvas );
}

void pressed_paste( GtkMenuItem *menu_item, gpointer user_data )
{
	paste_prepare();
	marq_x1 = mem_clip_x;
	marq_y1 = mem_clip_y;
	marq_x2 = mem_clip_x + mem_clip_w - 1;
	marq_y2 = mem_clip_y + mem_clip_h - 1;
	paste_init();
}

void pressed_paste_centre( GtkMenuItem *menu_item, gpointer user_data )
{
	int w, h;
	GtkAdjustment *hori, *vert;

	hori = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas));
	vert = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas));

	canvas_size(&w, &h);
	if (hori->page_size > w) mem_icx = 0.5;
	else mem_icx = (hori->value + hori->page_size * 0.5) / w;

	if (vert->page_size > h) mem_icy = 0.5;
	else mem_icy = (vert->value + vert->page_size * 0.5) / h;

	paste_prepare();
	align_size(can_zoom);
	marq_x1 = mem_width * mem_icx - mem_clip_w * 0.5;
	marq_y1 = mem_height * mem_icy - mem_clip_h * 0.5;
	marq_x2 = marq_x1 + mem_clip_w - 1;
	marq_y2 = marq_y1 + mem_clip_h - 1;
	paste_init();
}

static int copy_clip()
{
	int i, x, y, w, h, bpp, ofs, delta, len;


	x = marq_x1 < marq_x2 ? marq_x1 : marq_x2;
	y = marq_y1 < marq_y2 ? marq_y1 : marq_y2;
	w = abs(marq_x1 - marq_x2) + 1;
	h = abs(marq_y1 - marq_y2) + 1;

	bpp = MEM_BPP;
	bpp = 3;
	free(mem_clipboard);		// Lose old clipboard
	free(mem_clip_alpha);		// Lose old clipboard alpha
	mem_clip_mask_clear();		// Lose old clipboard mask
	mem_clip_alpha = NULL;
	if ((mem_channel == CHN_IMAGE) && mem_img[CHN_ALPHA] /*&&
		 !channel_dis[CHN_ALPHA]*/) mem_clip_alpha = malloc(w * h);
	mem_clipboard = malloc(w * h * bpp);
	text_paste = FALSE;

	if (!mem_clipboard)
	{
		free(mem_clip_alpha);
		alert_box_stock( _("Error"), _("Not enough memory to create clipboard"),
			GTK_STOCK_OK, NULL, NULL );
		return (FALSE);
	}

	mem_clip_bpp = bpp;
	mem_clip_x = x;
	mem_clip_y = y;
	mem_clip_w = w;
	mem_clip_h = h;

	/* Current channel */
	ofs = (y * mem_width + x) * bpp;
	delta = 0;
	len = w * bpp;

	for (i = 0; i < h; i++)
	{
		memcpy(mem_clipboard + delta, mem_img[mem_channel] + ofs, len);
		ofs += mem_width * bpp;
		delta += len;
	}

	return (TRUE);
}

static void cut_clip()
{
	int i, j, step;
	unsigned char *sel, fix = 255;

	spot_undo(UNDO_DRAW);
	step = mem_clip_mask ? 1 : 0;

	for (i = 0; i < mem_clip_h; i++)
	{
		sel = mem_clip_mask ? mem_clip_mask + i * mem_clip_w : &fix;
		for (j = 0; j < mem_clip_w; j++ , sel += step)
		{
			put_pixel(mem_clip_x + j, mem_clip_y + i);
		}
	}
}

void pressed_copy( GtkMenuItem *menu_item, gpointer user_data, gint item )
{
	if (!item && (tool_type == TOOL_SELECT) && (marq_status >= MARQUEE_PASTE))
	{
		mem_clip_x = marq_x1 < marq_x2 ? marq_x1 : marq_x2;
		mem_clip_y = marq_y1 < marq_y2 ? marq_y1 : marq_y2;
		return;
	}

	if (!copy_clip()) return;
	if (item) cut_clip();
	update_all_views();
	update_menus();
}


void update_menus()			// Update edit/undo menu
{
	if ( marq_status == MARQUEE_NONE )
	{
		men_item_state(menu_need_selection, FALSE);
		men_item_state(menu_crop, FALSE);
	}
	else
	{
		men_item_state(menu_need_marquee, TRUE);

		// If we are pasting disallow copy/cut/crop
		men_item_state(menu_need_selection, marq_status < MARQUEE_PASTE);

		// Only offer the crop option if the user hasn't selected everything
		men_item_state(menu_crop, (marq_status <= MARQUEE_DONE) &&
			((abs(marq_x1 - marq_x2) < mem_width - 1) ||
			(abs(marq_y1 - marq_y2) < mem_height - 1)));
	}

	// Forbid RGB-to-indexed paste, but allow indexed-to-RGB
	men_item_state(menu_need_clipboard, mem_clipboard && (mem_clip_bpp <= MEM_BPP));

	men_item_state(menu_undo, !!mem_undo_done);
	men_item_state(menu_redo, !!mem_undo_redo);
}

void canvas_undo_chores()
{
	int w, h;

	canvas_size(&w, &h);
	gtk_widget_set_usize(drawing_canvas, w, h);
	update_all_views();			// redraw canvas widget
	update_menus();
	init_pal();
}

void check_undo_paste_bpp()
{
	if (marq_status >= MARQUEE_PASTE && (mem_clip_bpp != MEM_BPP))
		pressed_select_none( NULL, NULL );
}

void main_undo( GtkMenuItem *menu_item, gpointer user_data )
{
	mem_undo_backward();
	check_undo_paste_bpp();
	canvas_undo_chores();
}

void main_redo( GtkMenuItem *menu_item, gpointer user_data )
{
	mem_undo_forward();
	check_undo_paste_bpp();
	canvas_undo_chores();
}


void update_cols()
{
	if (!mem_img[CHN_IMAGE]) return;	// Only do this if we have an image

	if ( marq_status >= MARQUEE_PASTE && text_paste )
	{
		render_text( drawing_canvas );
		check_marquee();
		gtk_widget_queue_draw( drawing_canvas );
	}
}

void init_pal()					// Initialise palette after loading
{
	update_cols();
}

void cleanse_txt( char *out, char *in )		// Cleans up non ASCII chars for GTK+2
{
	char *c;

	c = g_locale_to_utf8( (gchar *) in, -1, NULL, NULL, NULL );
	if ( c == NULL )
	{
		sprintf(out, "Error in cleanse_txt using g_*_to_utf8");
	}
	else
	{
		strcpy( out, c );
		g_free(c);
	}
}

void set_new_filename( char *fname )
{
	strncpy( mem_filename, fname, 250 );
	update_titlebar();
}

int do_a_load( char *fname )
{
	char mess[512], real_fname[300];
	int res, i;


	if ((fname[0] != DIR_SEP)
#ifdef WIN32
		&& (fname[1] != ':')
#endif
	)
	{
		getcwd(real_fname, 256);
		i = strlen(real_fname);
		real_fname[i] = DIR_SEP;
		real_fname[i + 1] = 0;
		strncat(real_fname, fname, 256);
	}
	else strncpy(real_fname, fname, 256);

	res = load_image(real_fname);

	if ( res<=0 )				// Error loading file
	{
		if (res == TOO_BIG)
		{
			snprintf(mess, 500, _("File is too big, must be <= to width=%i height=%i : %s"), MAX_WIDTH, MAX_HEIGHT, fname);
			alert_box_stock( _("Error"), mess, GTK_STOCK_OK, NULL, NULL );
		}
		else
		{
			alert_box_stock( _("Error"), _("Unable to load file"),
				GTK_STOCK_OK, NULL, NULL );
		}
		goto fail;
	}

	if ( res == FILE_MEM_ERROR )
	{
		memory_errors(1);		// Image was too large for OS
		res = -1;
	}

	/* Whether we loaded something or failed to, old image is gone anyway */
	register_file(real_fname);
	set_new_filename(real_fname);

	/* To prevent automatic paste following a file load when enabling
	 * "Changing tool commits paste" via preferences */
	pressed_select_none(NULL, NULL);
	reset_tools();
	update_all_views();

	gtk_adjustment_value_changed( gtk_scrolled_window_get_hadjustment(
		GTK_SCROLLED_WINDOW(scrolledwindow_canvas) ) );
	gtk_adjustment_value_changed( gtk_scrolled_window_get_vadjustment(
		GTK_SCROLLED_WINDOW(scrolledwindow_canvas) ) );
			// Set up scrollbars properly

//printf("new w=%i h=%i\n", mem_width, mem_height);

fail:
	return (res <= 0);
}



///	FILE SELECTION WINDOW

int check_file( char *fname )		// Does file already exist?  Ask if OK to overwrite
{
	char mess[512];

	if ( valid_file(fname) == 0 )
	{
		snprintf(mess, 500, _("File: %s already exists. Do you want to overwrite it?"), fname);
		if ( alert_box_stock( _("File Found"), mess, GTK_STOCK_NO, GTK_STOCK_YES, NULL ) != 2 ) return 1;
	}

	return 0;
}


static void change_image_format(GtkMenuItem *menuitem, GtkWidget *box)
{
	static int flags[] = {FF_COMPR, 0};
	GList *chain = GTK_BOX(box)->children->next->next;
	int i, ftype;

	ftype = (int)gtk_object_get_user_data(GTK_OBJECT(menuitem));
	/* Hide/show name/value widget pairs */
	for (i = 0; flags[i]; i++)
	{
		if (file_formats[ftype].flags & flags[i])
		{
			gtk_widget_show(((GtkBoxChild*)chain->data)->widget);
			gtk_widget_show(((GtkBoxChild*)chain->next->data)->widget);
		}
		else
		{
			gtk_widget_hide(((GtkBoxChild*)chain->data)->widget);
			gtk_widget_hide(((GtkBoxChild*)chain->next->data)->widget);
		}
		chain = chain->next->next;
	}
}

static void image_widgets(GtkWidget *box, char *name, int mode)
{
	GtkWidget *opt, *menu, *item, *label, *spin;
	int i, j, k, mask = FF_IDX;
	char *ext = strrchr(name, '.');

	ext = ext ? ext + 1 : "";
	switch (mode)
	{
	default: return;
	case FS_PNG_SAVE: mask = mem_img_bpp == 3 ? FF_RGB : mem_cols <= 2 ?
		FF_BW | FF_IDX : FF_IDX;
		break;
	}

	/* Create controls (!!! two widgets per value - used in traversal) */
	label = gtk_label_new(_("File Format"));
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 5);
	opt = gtk_option_menu_new();
	gtk_box_pack_start(GTK_BOX(box), opt, FALSE, FALSE, 5);

	label = gtk_label_new(_("JPEG Save Quality (100=High)"));
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 5);
	spin = add_a_spin(mem_jpeg_quality, 0, 100);
	gtk_box_pack_start(GTK_BOX(box), spin, FALSE, FALSE, 5);

	gtk_widget_show_all(box);
	menu = gtk_menu_new();
	for (i = j = k = 0; i < NUM_FTYPES; i++)
	{
		if (!(file_formats[i].flags & mask)) continue;
		if (!strncasecmp(ext, file_formats[i].ext, LONGEST_EXT) ||
			(file_formats[i].ext2[0] &&
			!strncasecmp(ext, file_formats[i].ext2, LONGEST_EXT)))
			j = k;
		item = gtk_menu_item_new_with_label(file_formats[i].name);
		gtk_object_set_user_data(GTK_OBJECT(item), (gpointer)i);
		gtk_signal_connect(GTK_OBJECT(item), "activate",
			GTK_SIGNAL_FUNC(change_image_format), (gpointer)box);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		k++;
  	}
	gtk_widget_show_all(menu);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(opt), menu);

	gtk_option_menu_set_history(GTK_OPTION_MENU(opt), j);

	gtk_signal_emit_by_name(GTK_OBJECT(g_list_nth_data(
		GTK_MENU_SHELL(menu)->children, j)), "activate", (gpointer)box);
}

static GtkWidget *ls_settings_box(char *name, int mode)
{
	GtkWidget *box;

	box = gtk_hbox_new(FALSE, 0);
	gtk_object_set_user_data(GTK_OBJECT(box), (gpointer)mode);

	switch (mode) /* Only save operations need settings */
	{
	case FS_PNG_SAVE:
		image_widgets(box, name, mode);
		break;
	default: /* Give a hidden empty box */
		return (box);
	}

	gtk_widget_show(box);
	return (box);
}

static int selected_file_type(GtkWidget *box)
{
	GtkWidget *opt;

	opt = BOX_CHILD(box, 1);
	opt = gtk_option_menu_get_menu(GTK_OPTION_MENU(opt));
	if (!opt) return (FT_NONE);
	opt = gtk_menu_get_active(GTK_MENU(opt));
	return ((int)gtk_object_get_user_data(GTK_OBJECT(opt)));
}

void init_ls_settings(ls_settings *settings, GtkWidget *box)
{
	int xmode;

	/* Set defaults */
	memset(settings, 0, sizeof(ls_settings));
	settings->ftype = FT_NONE;
	settings->jpeg_quality = mem_jpeg_quality;

	/* Read in settings */
	if (box)
	{
		xmode = (int)gtk_object_get_user_data(GTK_OBJECT(box));
		settings->mode = xmode;

		switch (xmode)
		{
		case FS_PNG_SAVE:
			settings->ftype = selected_file_type(box);
			settings->jpeg_quality = read_spin(BOX_CHILD(box, 3));
			break;
		default: /* Use defaults */
			break;
		}
	}
}

static void store_ls_settings(ls_settings *settings)
{
	guint32 fflags = file_formats[settings->ftype].flags;

	switch (settings->mode)
	{
	case FS_PNG_SAVE:
		if (fflags & FF_COMPR)
		{
			mem_jpeg_quality = settings->jpeg_quality;
		}
		break;
	}
}

static gboolean fs_destroy(GtkWidget *fs)
{
	gtk_window_set_transient_for(GTK_WINDOW(fs), NULL);
	gtk_widget_destroy(fs);

	if ( close_after_save ) delete_event( NULL, NULL, NULL);

	return FALSE;
}

static gint fs_ok(GtkWidget *fs)
{
	ls_settings settings;
	GtkWidget *xtra;
	char fname[256], *c, *ext, *ext2;
	int i, j;

	/* Pick up extra info */
	xtra = GTK_WIDGET(gtk_object_get_user_data(GTK_OBJECT(fs)));
	init_ls_settings(&settings, xtra);

	/* Needed to show progress in Windows GTK+2 */
//	gtk_window_set_modal(GTK_WINDOW(fs), FALSE);

	/* Looks better if no dialog under progressbar */
	gtk_widget_hide(fs);

	/* File extension */
	strncpy(fname, gtk_entry_get_text(GTK_ENTRY(
		GTK_FILE_SELECTION(fs)->selection_entry)), 256);
	c = strrchr(fname, '.');
	while (TRUE)
	{
		/* Cut the extension off */
		{
			ext = file_formats[settings.ftype].ext;
			if (!ext[0]) break;
		
			if (c) /* There is an extension */
			{
				/* Same extension? */
				if (!strncasecmp(c + 1, ext, 256)) break;
				/* Alternate extension? */
				ext2 = file_formats[settings.ftype].ext2;
				if (ext2[0] && !strncasecmp(c + 1, ext2, 256))
					break;
				/* Another file type? */
				for (i = 0; i < NUM_FTYPES; i++)
				{
					if (strncasecmp(c + 1, file_formats[i].ext, 256) &&
						strncasecmp(c + 1, file_formats[i].ext2, 256))
						continue;
					/* Truncate */
					*c = '\0';
					break;
				}
			}
			i = strlen(fname);
			j = strlen(ext);
			if (i + j + 1 > 250) break; /* Too long */
			fname[i] = '.';
			strncpy(fname + i + 1, ext, j + 1);
		}
		gtk_entry_set_text(GTK_ENTRY(
			GTK_FILE_SELECTION(fs)->selection_entry), fname);
		break;
	}

	/* Get filename the proper way */
	gtkncpy(fname, gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)), 250);

	switch (settings.mode)
	{
	case FS_PNG_LOAD:
		if (do_a_load(fname) == 1) goto redo;
		break;
	case FS_PNG_SAVE:
		if (check_file(fname)) goto redo;
		store_ls_settings(&settings);	// Update data in memory
		if (gui_save(fname, &settings) < 0) goto redo;
		set_new_filename(fname);
		update_all_views();	// Redraw in case transparency changed
		break;
	}

	update_menus();

	fs_destroy(fs);
//	gtk_window_set_transient_for(GTK_WINDOW(fs), NULL);
//	gtk_widget_destroy(fs);

	return FALSE;
redo:
	gtk_widget_show(fs);
//	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	return FALSE;
}


static char fs_last_dir[260];

void fs_set_dir(char *s)
{
	strncpy(fs_last_dir, s, 256);
}

void file_selector(int action_type)
{
	char *title = NULL, txt[300], txt2[300];
	GtkWidget *fs, *xtra;


	switch (action_type)
	{
	case FS_PNG_LOAD:
		title = _("Load Image File");
		if (check_for_changes(GTK_STOCK_NO) == 1) return;
		break;
	case FS_PNG_SAVE:
		title = _("Save Image File");
		break;
	}

	fs = gtk_file_selection_new(title);
	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_window_set_transient_for(GTK_WINDOW(fs), GTK_WINDOW(main_window));

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(fs_ok), GTK_OBJECT(fs));

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(fs_destroy), GTK_OBJECT(fs));

	gtk_signal_connect_object(GTK_OBJECT(fs),
		"delete_event", GTK_SIGNAL_FUNC(fs_destroy), GTK_OBJECT(fs));

	if ((action_type == FS_PNG_SAVE) && strcmp(mem_filename, _("Untitled")))
		strncpy( txt, mem_filename, 256 );	// If we have a filename and saving
	else
	{
		if ( strlen(fs_last_dir) == 0 ) txt[0]=0;	// Nothing set so leave empty
		else snprintf(txt, 256, "%s%c", fs_last_dir, DIR_SEP);
	}

	cleanse_txt( txt2, txt );		// Clean up non ASCII chars
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), txt2);

	xtra = ls_settings_box(txt2, action_type);
	gtk_box_pack_start(GTK_BOX(GTK_FILE_SELECTION(fs)->main_vbox), xtra,
		FALSE, TRUE, 0);
	gtk_object_set_user_data(GTK_OBJECT(fs), xtra);

	gtk_widget_show(fs);
//	gdk_window_raise(fs->window);	// Needed to ensure window is at the top
}

void align_size( float new_zoom )		// Set new zoom level
{
	GtkAdjustment *hori, *vert;
	int w, h, nv_h = 0, nv_v = 0;	// New positions of scrollbar

	if (zoom_flag) return;		// Needed as we could be called twice per iteration

	if (new_zoom < MIN_ZOOM) new_zoom = MIN_ZOOM;
	if (new_zoom > MAX_ZOOM) new_zoom = MAX_ZOOM;

	if (new_zoom == can_zoom) return;

	zoom_flag = 1;
	hori = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas));
	vert = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas));

	if (mem_ics == 0)
	{
		canvas_size(&w, &h);
		if (hori->page_size > w) mem_icx = 0.5;
		else mem_icx = (hori->value + hori->page_size * 0.5) / w;
		if (vert->page_size > h) mem_icy = 0.5;
		else mem_icy = (vert->value + vert->page_size * 0.5) / h;
	}
	mem_ics = 0;

	can_zoom = new_zoom;
	canvas_size(&w, &h);

	if (hori->page_size < w)
		nv_h = rint(w * mem_icx - hori->page_size * 0.5);

	if (vert->page_size < h)
		nv_v = rint(h * mem_icy - vert->page_size * 0.5);

	hori->value = nv_h;
	hori->upper = w;
	vert->value = nv_v;
	vert->upper = h;

	gtk_widget_set_usize(drawing_canvas, w, h);

	zoom_flag = 0;
	toolbar_zoom_update();
}

/* This tool is seamless: doesn't draw pixels twice if not requested to - WJ */
static void rec_continuous(int nx, int ny, int w, int h)
{
	linedata line1, line2, line3, line4;
	int ws2 = w >> 1, hs2 = h >> 1;
	int i, j, i2, j2, *xv;
	int dx[3] = {-ws2, w - ws2 - 1, -ws2};
	int dy[3] = {-hs2, h - hs2 - 1, -hs2};

	i = nx < tool_ox;
	j = ny < tool_oy;

	i2 = tool_ox + dx[i + 1] + 1 - i * 2;
	j2 = tool_oy + dy[j + 1] + 1 - j * 2;
	xv = &line3[0];

	if (tool_ox == nx)
	{
		line_init(line1, tool_ox + dx[i], j2,
			tool_ox + dx[i], ny + dy[j + 1]);
		line_init(line3, tool_ox + dx[i + 1], j2,
			tool_ox + dx[i + 1], ny + dy[j + 1]);
		line2[2] = line4[2] = -1;
	}
	else
	{
		line_init(line2, tool_ox + dx[i], tool_oy + dy[j + 1],
			nx + dx[i], ny + dy[j + 1]);
		line_nudge(line2, i2, j2);
		line_init(line3, tool_ox + dx[i + 1], tool_oy + dy[j],
			nx + dx[i + 1], ny + dy[j]);
		line_nudge(line3, i2, j2);
		line_init(line1, *xv, line3[1], *xv, line2[1]);
		line_init(line4, nx + dx[i + 1], ny + dy[j],
			nx + dx[i + 1], ny + dy[j + 1]);
	}

	draw_quad(line1, line2, line3, line4);
}

void update_all_views()				// Update whole canvas on all views
{
	if ( drawing_canvas ) gtk_widget_queue_draw( drawing_canvas );
}


void tool_action(int event, int x, int y, int button, gdouble pressure)
{
	int minx = -1, miny = -1, xw = -1, yh = -1;
	int i, j, k, rx, ry, /* k, rx, ry, sx, sy,*/ ts2, tr2, res;
	int ox, oy;
	gboolean rmb_tool, first_point = FALSE;

	/* Does tool draw with color B when right button pressed? */
	rmb_tool = (tool_type <= TOOL_SPRAY) || (tool_type == TOOL_FLOOD);

	if ( pen_down == 0 )
	{
		first_point = TRUE;
	}
	else if ( tool_ox == x && tool_oy == y ) return;	// Only do something with a new point

	ts2 = tool_size >> 1;
	tr2 = tool_size - ts2 - 1;

	/* Handle "exceptional" tools */
	res = 1;
	if ((tool_type == TOOL_SELECT) || (tool_type == TOOL_POLYGON))
	{
		if ( marq_status == MARQUEE_PASTE )		// User wants to drag the paste box
		{
			if ( x>=marq_x1 && x<=marq_x2 && y>=marq_y1 && y<=marq_y2 )
			{
				marq_status = MARQUEE_PASTE_DRAG;
				marq_drag_x = x - marq_x1;
				marq_drag_y = y - marq_y1;
			}
		}
		if ( marq_status == MARQUEE_PASTE_DRAG && ( button == 1 || button == 13 || button == 2 ) )
		{	// User wants to drag the paste box
			ox = marq_x1;
			oy = marq_y1;
			paint_marquee(0, x - marq_drag_x, y - marq_drag_y);
			marq_x1 = x - marq_drag_x;
			marq_y1 = y - marq_drag_y;
			marq_x2 = marq_x1 + mem_clip_w - 1;
			marq_y2 = marq_y1 + mem_clip_h - 1;
			paint_marquee(1, ox, oy);
		}
		if ( (marq_status == MARQUEE_PASTE_DRAG || marq_status == MARQUEE_PASTE ) &&
			(((button == 3) && (event == GDK_BUTTON_PRESS)) ||
			((button == 13) && (event == GDK_MOTION_NOTIFY))))
		{	// User wants to commit the paste
			commit_paste(TRUE);
		}
		if ( tool_type == TOOL_SELECT && button == 3 && (marq_status == MARQUEE_DONE ) )
		{
			pressed_select_none(NULL, NULL);
			set_cursor();
		}
		if ( tool_type == TOOL_SELECT && button == 1 && (marq_status == MARQUEE_NONE ||
			marq_status == MARQUEE_DONE) )		// Starting a selection
		{
			if ( marq_status == MARQUEE_DONE )
			{
				paint_marquee(0, marq_x1-mem_width, marq_y1-mem_height);
				i = close_to(x, y);
				if (!(i & 1) ^ (marq_x1 > marq_x2))
					marq_x1 = marq_x2;
				if (!(i & 2) ^ (marq_y1 > marq_y2))
					marq_y1 = marq_y2;
				set_cursor();
			}
			else
			{
				marq_x1 = x;
				marq_y1 = y;
			}
			marq_x2 = x;
			marq_y2 = y;
			marq_status = MARQUEE_SELECTING;
			paint_marquee(1, marq_x1-mem_width, marq_y1-mem_height);
		}
		else
		{
			if ( marq_status == MARQUEE_SELECTING )		// Continuing to make a selection
			{
				paint_marquee(0, marq_x1-mem_width, marq_y1-mem_height);
				marq_x2 = x;
				marq_y2 = y;
				paint_marquee(1, marq_x1-mem_width, marq_y1-mem_height);
			}
		}

	}
	else /* Some other kind of tool */
	{
		/* If proper button for tool */
		if ((button == 1) || ((button == 3) && rmb_tool))
		{
			// Do memory stuff for undo
			if (tool_type != TOOL_FLOOD) mem_undo_next(UNDO_TOOL);	
			res = 0; 
		}
	}

	/* Handle continuous mode */
	while (!res && mem_continuous && !first_point)
	{
		minx = tool_ox < x ? tool_ox : x;
		xw = (tool_ox > x ? tool_ox : x) - minx + tool_size;
		minx -= ts2;

		miny = tool_oy < y ? tool_oy : y;
		yh = (tool_oy > y ? tool_oy : y) - miny + tool_size;
		miny -= ts2;

		res = 1;

		if (ts2 ? tool_type == TOOL_SQUARE : tool_type < TOOL_SPRAY)
		{
			rec_continuous(x, y, tool_size, tool_size);
			break;
		}
		if (tool_type == TOOL_CIRCLE)
		{
			tline(tool_ox, tool_oy, x, y, tool_size);
			f_circle(x, y, tool_size);
			break;
		}
		xw = yh = -1; /* Nothing was done */
		res = 0; /* Non-continuous tool */
		break;
	}

	if (!res && (tool_type == TOOL_FLOOD))
	{
		/* Non-masked start point */
//		if (pixel_protected(x, y) < 255)
//		{
			j = get_pixel(x, y);
//			k = mem_channel != CHN_IMAGE ? channel_col_A[mem_channel] :
//				mem_img_bpp == 1 ? mem_col_A : PNG_2_INT(mem_col_A24);
			k = mem_img_bpp == 1 ? mem_col_A : PNG_2_INT(mem_col_A24);
			if (j != k) /* And never start on colour A */
			{
				spot_undo(UNDO_TOOL);
				flood_fill(x, y, j);
				update_all_views();
			}
//		}
		res = 1;
	}

	/* Handle non-continuous mode & tools */
	if (!res)
	{
		minx = x - ts2;
		miny = y - ts2;
		xw = tool_size;
		yh = tool_size;

		switch (tool_type)
		{
		case TOOL_SQUARE:
			f_rectangle(minx, miny, xw, yh);
			break;
		case TOOL_CIRCLE:
			f_circle(x, y, tool_size);
			break;
		case TOOL_SPRAY:
			for (j = 0; j < tool_flow; j++)
			{
				rx = x - ts2 + rand() % tool_size;
				ry = y - ts2 + rand() % tool_size;
				IF_IN_RANGE(rx, ry) put_pixel(rx, ry);
			}
			break;
		default: xw = yh = -1; /* Nothing was done */
			break;
		}
	}

	if ((xw >= 0) && (yh >= 0)) /* Some drawing action */
	{
		if (xw + minx > mem_width) xw = mem_width - minx;
		if (yh + miny > mem_height) yh = mem_height - miny;
		if (minx < 0) xw += minx , minx = 0;
		if (miny < 0) yh += miny , miny = 0;

		if ((xw >= 0) && (yh >= 0))
		{
			main_update_area(minx, miny, xw, yh);
		}
	}
	tool_ox = x;	// Remember the coords just used as they are needed in continuous mode
	tool_oy = y;
}

void check_marquee()		// Check marquee boundaries are OK - may be outside limits via arrow keys
{
	if ( marq_status >= MARQUEE_PASTE )
	{
		mtMAX( marq_x1, marq_x1, 1-mem_clip_w )
		mtMAX( marq_y1, marq_y1, 1-mem_clip_h )
		mtMIN( marq_x1, marq_x1, mem_width-1 )
		mtMIN( marq_y1, marq_y1, mem_height-1 )
		marq_x2 = marq_x1 + mem_clip_w - 1;
		marq_y2 = marq_y1 + mem_clip_h - 1;
		return;
	}
	/* Selection mode in operation */
	if ((marq_status != MARQUEE_NONE) /*|| (poly_status == POLY_DONE)*/)
	{
		mtMAX( marq_x1, marq_x1, 0 )
		mtMAX( marq_y1, marq_y1, 0 )
		mtMAX( marq_x2, marq_x2, 0 )
		mtMAX( marq_y2, marq_y2, 0 )
		mtMIN( marq_x1, marq_x1, mem_width-1 )
		mtMIN( marq_y1, marq_y1, mem_height-1 )
		mtMIN( marq_x2, marq_x2, mem_width-1 )
		mtMIN( marq_y2, marq_y2, mem_height-1 )
	}
}

static int vc_x1, vc_y1, vc_x2, vc_y2;	// Visible canvas

static void get_visible()
{
	GtkAdjustment *hori, *vert;

	hori = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );
	vert = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );

	vc_x1 = hori->value;
	vc_y1 = vert->value;
	vc_x2 = hori->value + hori->page_size - 1;
	vc_y2 = vert->value + vert->page_size - 1;
}

/* Clip area to visible canvas */
static void clip_area( int *rx, int *ry, int *rw, int *rh )
{
// !!! This assumes that if there's clipping, then there aren't margins
	if ( *rx<vc_x1 )
	{
		*rw = *rw + (*rx - vc_x1);
		*rx = vc_x1;
	}
	if ( *ry<vc_y1 )
	{
		*rh = *rh + (*ry - vc_y1);
		*ry = vc_y1;
	}
	if ( *rx + *rw > vc_x2 ) *rw = vc_x2 - *rx + 1;
	if ( *ry + *rh > vc_y2 ) *rh = vc_y2 - *ry + 1;
}

void update_paste_chunk( int x1, int y1, int x2, int y2 )
{
	int ux1, uy1, ux2, uy2, w, h;

	get_visible();
	canvas_size(&w, &h);

	ux1 = x1 > vc_x1 ? x1 : vc_x1;
	uy1 = y1 > vc_y1 ? y1 : vc_y1;
	ux2 = x2 < vc_x2 ? x2 : vc_x2;
	uy2 = y2 < vc_y2 ? y2 : vc_y2;
	if (ux2 >= w) ux2 = w - 1;
	if (uy2 >= h) uy2 = h - 1;

	/* Only repaint if on visible canvas */
	if ((ux1 <= ux2) && (uy1 <= uy2))
		repaint_paste(ux1, uy1, ux2, uy2);
}

void paint_marquee(int action, int new_x, int new_y)
{
	unsigned char *rgb;
	int x1, y1, x2, y2, w, h, new_x2, new_y2, mst, zoom = 1, scale = 1;
	int i, j, r, g, b, rx, ry, rw, rh, offx, offy;


	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	new_x2 = new_x + marq_x2 - marq_x1;
	new_y2 = new_y + marq_y2 - marq_y1;

	/* Get onscreen coords */
	check_marquee();
	x1 = (marq_x1 * scale) / zoom;
	y1 = (marq_y1 * scale) / zoom;
	x2 = (marq_x2 * scale) / zoom;
	y2 = (marq_y2 * scale) / zoom;
	w = abs(x2 - x1) + scale;
	h = abs(y2 - y1) + scale;
	if (x2 < x1) x1 = x2;
	if (y2 < y1) y1 = y2;

	get_visible();

	if (action == 0) /* Clear */
	{
		mst = marq_status;
		marq_status = 0;
		/* Redraw inner area if displaying the clipboard */
		if (mst >= MARQUEE_PASTE)
		{
			/* Do nothing if not moved anywhere */
			if ((new_x == marq_x1) && (new_y == marq_y1));
			/* Full redraw if no intersection */
			else if ((new_x2 < marq_x1) || (new_x > marq_x2) ||
				(new_y2 < marq_y1) || (new_y > marq_y2))
				repaint_canvas(margin_main_x + x1,
					margin_main_y + y1, w, h);
			/* Partial redraw */
			else
			{
				if (new_x != marq_x1) /* Horizontal shift */
				{
					ry = y1; rh = h;
					if (new_x < marq_x1) /* Move left */
					{
						rx = (new_x2 * scale) / zoom + scale;
						rw = x1 + w - rx;
					}
					else /* Move right */
					{
						rx = x1;
						rw = (new_x * scale) / zoom - x1;
					}
					clip_area(&rx, &ry, &rw, &rh);
					repaint_canvas(margin_main_x + rx,
						margin_main_y + ry, rw, rh);
				}
				if (new_y != marq_y1) /* Vertical shift */
				{
					rx = x1; rw = w;
					if (new_y < marq_y1) /* Move up */
					{
						ry = (new_y2 * scale) / zoom + scale;
						rh = y1 + h - ry;
					}
					else /* Move down */
					{
						ry = y1;
						rh = (new_y * scale) / zoom - y1;
					}
					clip_area(&rx, &ry, &rw, &rh);
					repaint_canvas(margin_main_x + rx,
						margin_main_y + ry, rw, rh);
				}
			}
		}
		/* Redraw only borders themselves */
		else
		{
			repaint_canvas(margin_main_x + x1,
				margin_main_y + y1 + 1, 1, h - 2);
			repaint_canvas(margin_main_x + x1 + w - 1,
				margin_main_y + y1 + 1, 1, h - 2);
			repaint_canvas(margin_main_x + x1,
				margin_main_y + y1, w, 1);
			repaint_canvas(margin_main_x + x1,
				margin_main_y + y1 + h - 1, w, 1);
		}
		marq_status = mst;
	}

	/* Draw */
	else if ((action == 1) || (action == 11))
	{
		r = 255; g = b = 0; /* Draw in red */
		if (marq_status >= MARQUEE_PASTE)
		{
			/* Display paste RGB, only if not being called from repaint_canvas */
			/* Only do something if there is a change in position */
			if (action == 1 &&
				((new_x != marq_x1) || (new_y != marq_y1)))
				update_paste_chunk(marq_x1 < 0 ? 0 : x1 + 1,
					marq_y1 < 0 ? 0 : y1 + 1,
					x1 + w - 2, y1 + h - 2);
			r = g = 0; b = 255; /* Draw in blue */
		}

		/* Determine visible area */
		rx = x1; ry = y1; rw = w; rh = h;
		clip_area(&rx, &ry, &rw, &rh);
		if ((rw < 1) || (rh < 1)) return;

		offx = (abs(rx - x1) % 6) * 3;
		offy = (abs(ry - y1) % 6) * 3;

		/* Create pattern */
		j = (rw > rh ? rw : rh) * 3 + 6 * 3; /* 6 pixels for offset */
		rgb = malloc(j + 2 * 3); /* 2 extra pixels reserved for loop */
		if (!rgb) return;
		memset(rgb, 255, j);
		for (i = 0; i < j; i += 6 * 3)
		{
			rgb[i + 0] = rgb[i + 3] = rgb[i + 6] = r;
			rgb[i + 1] = rgb[i + 4] = rgb[i + 7] = g;
			rgb[i + 2] = rgb[i + 5] = rgb[i + 8] = b;
		}

		i = ((mem_width + zoom - 1) * scale) / zoom;
		j = ((mem_height + zoom - 1) * scale) / zoom;
		if (rx + rw > i) rw = i - rx;
		if (ry + rh > j) rh = j - ry;

		if ((x1 >= vc_x1) && (marq_x1 >= 0) && (marq_x2 >= 0))
			gdk_draw_rgb_image(drawing_canvas->window,
				drawing_canvas->style->black_gc,
				margin_main_x + rx, margin_main_y + ry,
				1, rh, GDK_RGB_DITHER_NONE, rgb + offy, 3);

		if ((x1 + w - 1 <= vc_x2) && (marq_x1 < mem_width) && (marq_x2 < mem_width))
			gdk_draw_rgb_image(drawing_canvas->window,
				drawing_canvas->style->black_gc,
				margin_main_x + rx + rw - 1, margin_main_y + ry,
				1, rh, GDK_RGB_DITHER_NONE, rgb + offy, 3);

		if ((y1 >= vc_y1) && (marq_y1 >= 0) && (marq_y2 >= 0))
			gdk_draw_rgb_image(drawing_canvas->window,
				drawing_canvas->style->black_gc,
				margin_main_x + rx, margin_main_y + ry,
				rw, 1, GDK_RGB_DITHER_NONE, rgb + offx, 0);

		if ((y1 + h - 1 <= vc_y2) && (marq_y1 < mem_height) && (marq_y2 < mem_height))
			gdk_draw_rgb_image(drawing_canvas->window,
				drawing_canvas->style->black_gc,
				margin_main_x + rx, margin_main_y + ry + rh - 1,
				rw, 1, GDK_RGB_DITHER_NONE, rgb + offx, 0);

		free(rgb);
	}
}


int close_to( int x1, int y1 )		// Which corner of selection is coordinate closest to?
{
	return ((x1 + x1 <= marq_x1 + marq_x2 ? 0 : 1) +
		(y1 + y1 <= marq_y1 + marq_y2 ? 0 : 2));
}


void register_file( char *filename )		// Called after successful load/save
{
	char txt[280], *c;

	c = strrchr( filename, DIR_SEP );
	if (c)
	{
		txt[0] = *c;
		*c = '\0';		// Strip off filename
		snprintf(fs_last_dir, 256, "%s", filename);
		*c = txt[0];
	}
}

void create_default_image()			// Create default new image
{
	int	nw = DEFAULT_WIDTH,
		nh = DEFAULT_HEIGHT;

	do_new_one( nw, nh );
}
